package com.tbyf.his.web.controller.emport;

import com.alibaba.druid.pool.DruidDataSource;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.emport.domain.param.*;
import com.tbyf.his.emport.entity.ExcelRule;
import com.tbyf.his.emport.entity.ExcelTemplate;
import com.tbyf.his.emport.service.ExcelFieldRuleService;
import com.tbyf.his.emport.service.ExcelRuleService;
import com.tbyf.his.emport.service.ExcelTemplateFieldService;
import com.tbyf.his.emport.service.ExcelTemplateService;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/1/10 15:52
 */
@RestController
@Api(tags = "excel导入接口")
@RequestMapping("/emport")
@Slf4j
public class EmportController {

    @Autowired
    private ExcelTemplateService excelTemplateService;

    @Autowired
    private ExcelTemplateFieldService excelTemplateFieldService;

    @Autowired
    private ExcelRuleService excelRuleService;

    @Autowired
    private ExcelFieldRuleService excelFieldRuleService;

    /*===================模板接口====================*/

    @GetMapping("/template")
    @ApiOperation("分页模板查询")
    public TableDataInfo queryTemplate(@Validated QueryTemplateParam param) {
        return excelTemplateService.queryTemplate(param);
    }

    @GetMapping("/template/download")
    @ApiOperation("模板下载")
    public void downloadTemplate(@RequestParam Long templateId, HttpServletResponse response) {
        excelTemplateService.downloadTemplate(templateId, response);
    }

    @PostMapping("/template")
    @ApiOperation("新增模板信息")
    public AjaxResult addTemplate(@RequestBody @Validated ExcelTemplate template) {
        boolean result = excelTemplateService.addTemplate(template);
        return AjaxResult.success();
    }

    @PutMapping("/template")
    @ApiOperation("修改模板信息")
    public AjaxResult updateTemplate(@RequestBody @Validated ExcelTemplate template) {
        if (Objects.isNull(template.getId())) {
            return AjaxResult.error("id不能为空");
        }
        boolean result = excelTemplateService.updateTemplate(template);
        return AjaxResult.success();
    }

    @DeleteMapping("/template")
    @ApiOperation("删除模板信息")
    public AjaxResult deleteTemplate(@RequestParam Long templateId) {
        boolean result = excelTemplateService.deleteTemplate(templateId);
        return AjaxResult.success();
    }

    /*===================模板字段接口====================*/

    @GetMapping("/template/field")
    @ApiOperation("分页模板字段查询")
    public TableDataInfo queryTemplateField(@Validated QueryTemplateFieldParam param) {
        return excelTemplateFieldService.queryTemplateField(param);
    }

    @DeleteMapping("/template/field")
    @ApiOperation("删除模板字段")
    public AjaxResult deleteTemplateField(@RequestParam Long templateFieldId) {
        boolean result = excelTemplateFieldService.deleteTemplateField(templateFieldId);
        return AjaxResult.success();
    }

    @PostMapping("/template/field")
    @ApiOperation("新增模板字段配置")
    public AjaxResult addTemplateField(@RequestBody @Validated AddExcelTemplateFieldParam param) {
        boolean result = excelTemplateFieldService.addTemplateField(param);
        return AjaxResult.success();
    }

    @PutMapping("/template/field")
    @ApiOperation("修改模板字段配置")
    public AjaxResult updateTemplateField(@RequestBody @Validated UpdateExcelTemplateFieldParam param) {
        boolean result = excelTemplateFieldService.updateTemplateField(param);
        return AjaxResult.success();
    }

    @GetMapping("/template/field/test")
    @ApiOperation("模板字段规则测试")
    public AjaxResult fieldRuleTest(@RequestParam String fieldId, @RequestParam String value) {
        return excelTemplateFieldService.fieldRuleTest(fieldId, value);
    }

    /*===================规则接口====================*/

    @GetMapping("/rule")
    @ApiOperation("规则信息查询")
    public TableDataInfo queryRule(@Validated QueryExcelRuleParam param) {
        return excelRuleService.queryRule(param);
    }

    @GetMapping("/rule/dict")
    @ApiOperation("规则字典查询")
    public AjaxResult queryRuleDict() {
        return AjaxResult.success(excelRuleService.queryRuleDict());
    }

    @DeleteMapping("/rule")
    @ApiOperation("删除规则")
    public AjaxResult deleteRule(@RequestParam Long ruleId) {
        boolean result = excelRuleService.deleteRule(ruleId);
        return AjaxResult.success();
    }

    @PostMapping("/rule")
    @ApiOperation("新增规则配置")
    public AjaxResult addRule(@RequestBody @Validated AddExcelRuleParam param) {
        boolean result = excelRuleService.addRule(param);
        return AjaxResult.success();
    }

    @PutMapping("/rule")
    @ApiOperation("修改规则配置")
    public AjaxResult updateRule(@RequestBody @Validated UpdateExcelRuleParam param) {
        boolean result = excelRuleService.updateRule(param);
        return AjaxResult.success();
    }

    @PostMapping("/rule/test")
    @ApiOperation("规则测试接口")
    public AjaxResult testRule(@RequestBody @Validated TestRuleParam param) {
        return excelRuleService.testRule(param);
    }

    @GetMapping("/rule/name")
    @ApiOperation("查询规则名称集合")
    public AjaxResult getRuleNames(@RequestParam String ids) {
        return AjaxResult.success(excelRuleService.getRuleNames(ids));
    }

    /*===================模板规则接口====================*/

    @PostMapping("/fieldRule")
    @ApiOperation("配置字段规则绑定")
    public AjaxResult bindRule(@RequestBody @Validated BindRulesParam param) {
        boolean result = excelFieldRuleService.bindRules(param);
        return AjaxResult.success();
    }

    @GetMapping("/fieldRule")
    @ApiOperation("查询模板绑定规则")
    public AjaxResult listFieldRule(@RequestParam String fieldId) {
        List<ExcelRule> result = excelFieldRuleService.listFieldRule(fieldId);
        return AjaxResult.success(result);
    }

    @PostMapping("/upload")
    @ApiOperation("导入excel")
    public AjaxResult upload(@Validated UploadExcelParam param) {
        excelTemplateService.importExcel(param);
        return AjaxResult.success();
    }

    @PostMapping("/initTemplate")
    @ApiOperation("模板初始化")
    public AjaxResult initTemplate(@Validated UploadExcelParam param) {
        excelTemplateService.initTemplate(param);
        return AjaxResult.success();
    }

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @GetMapping("/db/table")
    @ApiOperation("获取指定数据库中所有的表")
    public AjaxResult getDbTable(@RequestParam String dataSourceId) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = "select  TABLE_NAME AS NAME,COMMENTS AS COMMENTS from user_tab_comments WHERE TABLE_TYPE = 'TABLE'";
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = "select  TABLE_NAME AS NAME ,table_comment AS COMMENTS from information_schema.tables where  table_type='BASE TABLE'";
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps.stream().filter(
                    distinctByKey(b -> b.get("NAME"))
            ).distinct().collect(Collectors.toList()));
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }

    @GetMapping("/db/field")
    @ApiOperation("获取指定表的字段信息")
    public AjaxResult getDbField(@RequestParam String dataSourceId,
                                 @RequestParam String tableName) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
            String sql = StrFormatter.format("SELECT UTC.COLUMN_NAME AS CNAME,UTC.DATA_TYPE AS CTYPE,( SELECT  UCC.COMMENTS  FROM user_col_comments UCC WHERE UCC.COLUMN_NAME = UTC.COLUMN_NAME AND UCC.TABLE_NAME = UTC.TABLE_NAME ) AS CCOMMENT FROM user_tab_columns UTC WHERE UTC.TABLE_NAME = '{}'", tableName);
            if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
                sql = StrFormatter.format("SELECT column_name AS CNAME, DATA_TYPE AS CTYPE, column_comment AS CCOMMENT FROM information_schema.COLUMNS WHERE table_name = '{}'", tableName);
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            return AjaxResult.success(maps);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }


    public static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }


}
