package com.tbyf.his.adapter.task;

import com.tbyf.his.adapter.domain.PublishColumnDto;
import com.tbyf.his.adapter.domain.ValueDomainDto;
import com.tbyf.his.adapter.mapper.AdapterPublishTaskMapper;
import com.tbyf.his.adapter.util.AdapterConstant;
import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.core.redis.RedisCache;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.utils.DateUtils;
import com.tbyf.his.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 数据适配相关方法
 *
 * @author lzz
 * @date 2022/12/9 14:01
 */

@Service
@Slf4j
@DataSource(DataSourceType.SLAVE)
public class DataAdapterService {

    public static final String XML_DOCUMENT = "<{}>{}</{}>";

    public static final String PARAMETERS = "<parameters><sourcetype>{}</sourcetype><sourceid>{}</sourceid><datasetcode>{}</datasetcode></parameters>";

    @Autowired
    private AdapterPublishTaskMapper adapterPublishTaskMapper;

    @Autowired
    private AdapterProperties adapterProperties;

    @Autowired
    private RedisCache redisCache;

    /**
     * 将map数据进行字段格式转换
     *
     * @param dataList
     * @param columnList
     * @return
     */
    public List<Map<String, String>> columnConvert(List<Map<String, String>> dataList, List<PublishColumnDto> columnList) {
        if (CollectionUtils.isEmpty(dataList) || CollectionUtils.isEmpty(columnList)) {
            return Collections.emptyList();
        }
        // 转换后的数据
        List<Map<String, String>> convertDataList = new ArrayList<>();
        // 列名格式对应
        final Map<String, PublishColumnDto> columnDtoMap = parsePublicColumn(dataList.get(0), columnList);
        dataList.forEach(item -> {
            Map<String, String> map = new HashMap<>();
            item.forEach((key, value) -> {
                final PublishColumnDto columnDto = columnDtoMap.get(key);
                if (columnDto != null) {
                    map.put(columnDto.getDataSetField(), convertData(value, columnDto));
                }
            });
            convertDataList.add(map);
        });
        return convertDataList;
    }

    /**
     * 将map转换成xml格式
     *
     * @param dataList
     * @param columnList
     * @return
     */
    public List<String> columnConvertByXml(List<Map<String, String>> dataList, List<PublishColumnDto> columnList) {
        if (CollectionUtils.isEmpty(dataList) || CollectionUtils.isEmpty(columnList)) {
            return Collections.emptyList();
        }
        // 转换后的数据
        List<String> xmlList = new ArrayList<>();
        // 列名格式对应
        final Map<String, PublishColumnDto> columnDtoMap = parsePublicColumn(dataList.get(0), columnList);
        // 数据集代码确定
        String dataSetCode = columnList.get(0).getDataSetCode();
        dataList.forEach(item -> {
            StringBuilder sb = new StringBuilder("<Program>");
            sb.append(StrFormatter.format(PARAMETERS, "I", adapterProperties.getSourceId(), dataSetCode));
            Map<String, StringBuilder> map = new HashMap<>();
            item.forEach((key, value) -> {
                final PublishColumnDto columnDto = columnDtoMap.get(key);
                if (columnDto != null) {
                    String dataDocument = StrFormatter.format(XML_DOCUMENT, columnDto.getDataSetField(), convertData(value, columnDto), columnDto.getDataSetField());
                    StringBuilder builder = map.get(columnDto.getDataSetCode());
                    if (builder == null) {
                        builder = new StringBuilder();
                        map.put(columnDto.getDataSetCode(), builder);
                    }
                    builder.append(dataDocument);
                }
            });
            map.forEach((code, sbs) -> {
                sb.append(StrFormatter.format(XML_DOCUMENT, code,
                        StrFormatter.format(XML_DOCUMENT, "row", sbs.toString(), "row"),
                        code));
            });
            sb.append("</Program>");
            System.out.println(sb);
            xmlList.add(sb.toString());
        });
        return xmlList;
    }

    /**
     * 转换成字段名-发布列的格式,避免出现查询字段大小写的问题,选择第一条查询数据进行转换
     *
     * @param map
     * @param columnList
     * @return
     */
    public Map<String, PublishColumnDto> parsePublicColumn(Map<String, String> map, List<PublishColumnDto> columnList) {
        Map<String, PublishColumnDto> dtoMap = new HashMap<>();
        map.forEach((key, value) -> {
            final Optional<PublishColumnDto> optional = columnList.stream().filter(c -> StringUtils.equalsIgnoreCase(key, c.getSourceField())).findFirst();
            optional.ifPresent(dto -> {
                dtoMap.put(key, dto);
            });
        });
        return dtoMap;
    }

    /**
     * 字段数据转换
     * 目前数据转换的模式有: 值转换,常量转换,公式转换,sql转换,基础数据
     * 目前常量转换,公式转换,sql转换,基础数据转换均由任务生成sql的时候预设
     * 这边只需要考虑值转换的类型与值域判断
     * 空值默认返回空串
     *
     * @return
     */
    public String convertData(String data, PublishColumnDto columnDto) {
        if (StringUtils.isBlank(data)) {
            return "";
        }
        switch (columnDto.getConvertMode()) {
            // 值转换
            case AdapterConstant.RANGE_CONVERT:
                if (StringUtils.isNotEmpty(columnDto.getDomainCode())) {
                    // 走值域代码的模式
                    String key = columnDto.getDomainCode() + ":" + data;
                    if (!redisCache.redisTemplate.hasKey(key)) {
                        ValueDomainDto domainDtos = adapterPublishTaskMapper.selectValueDomainList(columnDto.getDomainCode(), data);
                        if (domainDtos != null && StringUtils.isNotBlank(domainDtos.getCode())) {
                            redisCache.setCacheObject(key, domainDtos.getCode());
                            return domainDtos.getCode();
                        } else {
                            redisCache.setCacheObject(key, "");
                            return "";
                        }
                    } else {
                        return redisCache.getCacheObject(key);
                    }
                } else {
                    // 走普通列转换的模式
                    return convertValue(data, columnDto);
                }
            default:
                return convertValue(data, columnDto);
        }
    }

    /**
     * 数据类型转换
     * 源数据有String,Int,Date等三种格式
     *
     * @param data
     * @param columnDto
     * @return
     */
    public String convertValue(String data, PublishColumnDto columnDto) {
        /**
         * < 和 & 转译为 &lt; 和 &amp;
         * 在xml解析时只有< 和 & 是非法的, 其它都是可以合法存在的, 但把它们都进行转义是一个好的习惯
         */
        if (!StringUtils.isEmpty(data)) {
            data = data
                    .replaceAll("<", "&lt;")
                    .replaceAll(">", "&gt;")
                    .replaceAll("&", "&amp;")
                    .replaceAll("'", "&apos;")
                    .replaceAll("\"", "&quot;")
                    .replaceAll("©", "&apos;");
        }

        final String type = columnDto.getDataSetType().toUpperCase();

        if (type.contains("CHAR") || type.contains("CLOB")) {
            return data;
        } else if (type.contains("DATE") || type.contains("DATETIME")) {
            return DateUtils.parseDateToStr("yyyy-MM-dd HH:mm:ss", DateUtils.parseDate(data));
        } else if (type.contains("NUMBER")) {
            return data;
        }
        return data;
    }


    public static String doSoap(String postUrl, String soapXml) {
        String retStr = "";
        // 创建HttpClientBuilder
        HttpClientBuilder httpClientBuilder = HttpClientBuilder.create();
        // HttpClient
        CloseableHttpClient closeableHttpClient = httpClientBuilder.build();
        HttpPost httpPost = new HttpPost(postUrl);
        // 设置请求和传输超时时间
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(20000).setConnectTimeout(20000).build();
        httpPost.setConfig(requestConfig);
        try {
            httpPost.setHeader("Content-Type", "text/xml; charset=utf-8");
            StringEntity data = new StringEntity(soapXml, StandardCharsets.UTF_8);
            httpPost.setEntity(data);
            CloseableHttpResponse response = closeableHttpClient.execute(httpPost);
            HttpEntity httpEntity = response.getEntity();
            if (httpEntity != null) {
                // 打印响应内容
                retStr = EntityUtils.toString(httpEntity, "UTF-8");
            }
        } catch (Exception e) {
            log.error("调用ws异常", e);
        } finally {
            try {
                // 释放资源
                closeableHttpClient.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return retStr;
    }

    /**
     * xml返回值校验
     *
     * @param xmlResult
     */
    public static void verifyXmlResult(String xmlResult) {
        final String aReturn = getXmlText(xmlResult, "return");
        if (StringUtils.isBlank(aReturn)) {
            throw new RuntimeException(xmlResult);
        }
        final String errorNo = getXmlText(aReturn, "ErrorNo");
        if (!"0".equals(errorNo)) {
            throw new RuntimeException(aReturn);
        }
    }

    public static String getXmlText(String xml, String nodeName) {
        final Document doc;
        try {
            doc = DocumentHelper.parseText(xml);
        } catch (Exception e) {
            return null;
        }
        return getXmlText(doc.getRootElement(), nodeName);
    }

    /**
     * 获取指定节点的值
     *
     * @param element
     * @param nodeName
     * @return
     */
    public static String getXmlText(Element element, String nodeName) {
        final Iterator<Element> iterator = element.elementIterator();
        while (iterator.hasNext()) {
            final Element next = iterator.next();
            if (next.getName().equalsIgnoreCase(nodeName)) {
                return next.getTextTrim();
            }
            final String text = getXmlText(next, nodeName);
            if (StringUtils.isNotBlank(text)) {
                return text;
            }
        }
        return null;
    }


}
