package com.tbyf.his.emport.core;

import com.alibaba.druid.pool.DruidDataSource;
import com.google.common.collect.Lists;
import com.tbyf.his.common.exception.base.BaseException;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.emport.domain.vo.FieldInfoVO;
import com.tbyf.his.framework.datasource.DynamicDataSourceContextHolder;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.springframework.util.CollectionUtils;

import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 动态数据导入工具类
 *
 * @author lzz
 * @date 2023/2/3 9:31
 */

public class EmportUtils {

    /**
     * 查询这一行是否包含此数据
     *
     * @param row
     * @param str 支持正则包裹的表达式
     * @return
     */
    public static boolean findText(Row row, String str) {
        final Iterator<Cell> cellIterator = row.cellIterator();
        while (cellIterator.hasNext()) {
            final Cell next = cellIterator.next();
            final String value = getValue(next);
            if (StringUtils.isNotBlank(value) && Pattern.matches(str, value)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 哪一行开始有数据
     *
     * @param sheet
     * @return
     */
    public static int hasText(Sheet sheet) {
        for (Row row : sheet) {
            for (Cell cell : row) {
                if (StringUtils.isNotBlank(getValue(cell))) {
                    return cell.getRowIndex();
                }
            }
        }
        return -1;
    }

    /**
     * 从单元格获取数据
     *
     * @param cell
     * @return
     */
    public static String getValue(Cell cell) {
        final CellType cellType = cell.getCellType();
        switch (cellType) {
            case STRING:
                return cell.getStringCellValue().trim();
            case BOOLEAN:
                return String.valueOf(cell.getBooleanCellValue());
            case NUMERIC:
                return ((XSSFCell) cell).getCTCell().getV().trim();
            default:
                return "";
        }
    }

    /**
     * 从单元格获取数据
     *
     * @param cell
     * @return
     */
    public static String getValue(Cell cell, String type) {
        final CellType cellType = cell.getCellType();
        switch (cellType) {
            case STRING:
                return cell.getStringCellValue().trim();
            case BOOLEAN:
                return String.valueOf(cell.getBooleanCellValue());
            case NUMERIC:
                if ("Number".equals(type)) {
                    return String.valueOf(cell.getNumericCellValue());
                } else if ("Date(YYYY-MM-DD)".equals(type)) {
                    return cell.getLocalDateTimeCellValue().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
                } else if ("Date(YYYY-MM-DD HH:mm:ss)".equals(type)) {
                    return cell.getLocalDateTimeCellValue().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
                }
                return ((XSSFCell) cell).getCTCell().getV();
            default:
                return "";
        }
    }

    /**
     * 生成批量sql
     * oracle:
     * * INSERT INTO TEST_USER (NAME,AGE)
     * * SELECT '1','2' FROM DUAL
     * * UNION ALL
     * * SELECT '1','2' FROM DUAL
     *
     * @param tableName
     * @param dataSourceId
     * @param fieldList
     * @param dataList
     * @param batchSize
     * @return
     */
    public static List<String> getBatchSql(String tableName,
                                           String dataSourceId,
                                           List<FieldInfoVO> fieldList,
                                           List<List<String>> dataList,
                                           int batchSize) {
        final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
        List<String> sqlList = new ArrayList<>();
        // 计算切分次数
        // int limit = (dataList.size() + batchSize - 1) / batchSize;
        final List<List<List<String>>> partition = Lists.partition(dataList, batchSize);
        if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
            // INSERT INTO TABLE_NAME(FIELD1,FIELD2)VALUES('1','2'),('3','4')
            String sql = "INSERT INTO " + tableName + "("
                    + fieldList.stream().map(FieldInfoVO::getDataFieldName).collect(Collectors.joining(","))
                    + ") VALUES";
            for (List<List<String>> part : partition) {
                StringBuilder sb = new StringBuilder(sql);
                for (List<String> row : part) {
                    sb.append("(");
                    for (int i = 0; i < row.size(); i++) {
                        sb.append(convertMysqlValue(fieldList.get(i).getDataType(), row.get(i))).append(",");
                    }
                    sb.deleteCharAt(sb.length() - 1);
                    sb.append("),");
                }
                sb.deleteCharAt(sb.length() - 1);
                sqlList.add(sb.toString());
            }
        } else {
            String sql = "INSERT INTO " + tableName + "("
                    + fieldList.stream().map(FieldInfoVO::getDataFieldName).collect(Collectors.joining(","))
                    + ")";
            for (List<List<String>> part : partition) {
                StringBuilder sb = new StringBuilder(sql);
                for (List<String> row : part) {
                    sb.append(" SELECT ");
                    for (int i = 0; i < row.size(); i++) {
                        sb.append(convertOracleValue(fieldList.get(i).getDataType(), row.get(i))).append(",");
                    }
                    sb.deleteCharAt(sb.length() - 1);
                    sb.append(" FROM DUAL UNION ALL ");
                }
                sb.deleteCharAt(sb.length() - 1);
                sqlList.add(sb.toString());
            }
        }
        return sqlList;
    }

    /**
     * 生成单行数据的插入语句
     *
     * @param tableName
     * @param dataSourceId
     * @param fieldList
     * @param dataList
     * @return
     */
    public static String getInsertSql(String tableName,
                                      String dataSourceId,
                                      List<FieldInfoVO> fieldList,
                                      List<String> dataList) {
        final DruidDataSource dataSource = (DruidDataSource) DynamicDataSourceContextHolder.dataSourcesMap.get(dataSourceId);
        if (dataSource.getDriverClassName().toLowerCase().contains("mysql")) {
            StringBuilder sb = new StringBuilder();
            sb.append("INSERT INTO ")
                    .append(tableName)
                    .append("(")
                    .append(fieldList.stream().map(FieldInfoVO::getDataFieldName).collect(Collectors.joining(",")))
                    .append(") VALUES(");
            for (int i = 0; i < dataList.size(); i++) {
                sb.append(convertMysqlValue(fieldList.get(i).getDataType(), dataList.get(i))).append(",");
            }
            sb.deleteCharAt(sb.length() - 1);
            sb.append(")");
            return sb.toString();
        } else {
            StringBuilder sb = new StringBuilder();
            sb.append("INSERT INTO ")
                    .append(tableName)
                    .append("(")
                    .append(fieldList.stream().map(FieldInfoVO::getDataFieldName).collect(Collectors.joining(",")))
                    .append(") VALUES(");
            for (int i = 0; i < dataList.size(); i++) {
                sb.append(convertOracleValue(fieldList.get(i).getDataType(), dataList.get(i))).append(",");
            }
            sb.deleteCharAt(sb.length() - 1);
            sb.append(")");
            return sb.toString();
        }
    }

    public static String convertOracleValue(String type, String value) {
        if (StringUtils.isEmpty(type)) {
            return "'" + value + "'";
        }
        switch (type) {
            case "Date(YYYY-MM-DD)":
                return "to_date('" + value + "','YYYY-MM-DD')";
            case "Date(YYYY-MM-DD HH:mm:ss)":
                return "to_date('" + value + "','YYYY-MM-DD HH24:MI:SS')";
            default:
                return "'" + value + "'";
        }
    }

    public static String convertMysqlValue(String type, String value) {
        if (StringUtils.isEmpty(type)) {
            return "'" + value + "'";
        }
        switch (type) {
            case "Date(YYYY-MM-DD)":
                return "STR_TO_DATE('" + value + "','%Y-%m-%d') ";
            case "Date(YYYY-MM-DD HH:mm:ss)":
                return "STR_TO_DATE('" + value + "','%Y-%m-%d %H:%i:%s') ";
            default:
                return "'" + value + "'";
        }
    }

    /**
     * 数据批量校验
     *
     * @param ruleList
     * @param value
     */
    public static void validate(List<RuleContent> ruleList, String value) {
        if (!CollectionUtils.isEmpty(ruleList)) {
            if (StringUtils.isBlank(value)) {
                throw new BaseException("数据不能为空");

            }
            for (RuleContent rule : ruleList) {
                final RuleValidator validator = EmportConfig.getValidator(rule.getMode());
                if (validator == null) {
                    throw new EmportException(rule, "没有找到对应的规则校验器:{}", rule.getMode());
                }
                validator.validate(rule);
            }
        }
    }

    /**
     * 获取横纵坐标
     *
     * @return
     */
    public static Integer getHorizontal(String address) {
        final String s = address.split(",")[0];
        try {
            return Integer.parseInt(s);
        } catch (Exception e) {
            return letterToNumber(s);
        }
    }

    /**
     * 字母转数字
     *
     * @param letter
     * @return
     */
    public static int letterToNumber(String letter) {
        int length = letter.length();
        int num = 0;
        int number = 0;
        for (int i = 0; i < length; i++) {
            char ch = letter.charAt(length - i - 1);
            num = (int) (ch - 'A' + 1);
            num *= Math.pow(26, i);
            number += num;
        }
        return number;
    }

    /**
     * 获取对应的横纵坐标
     * 通过代码与标题
     * 代码就是横坐标+1 标题就是+2
     *
     * @param sheet
     * @param fieldInfo
     * @return
     */
    public static String findAddress(Sheet sheet, FieldInfoVO fieldInfo) {
        int allRows = sheet.getLastRowNum() + 1;
        for (int a = 0; a < allRows; a++) {
            for (Cell cell : sheet.getRow(a)) {
                final String value = getValue(cell);
                if (StringUtils.isNotBlank(fieldInfo.getFieldName()) && StringUtils.equals(value, fieldInfo.getFieldName())) {
                    return cell.getColumnIndex() + 2 + "," + cell.getRowIndex();
                }
                if (StringUtils.isNotBlank(fieldInfo.getFieldTitle()) && StringUtils.equals(value, fieldInfo.getFieldTitle())) {
                    return cell.getColumnIndex() + 1 + "," + cell.getRowIndex();
                }
            }

        }
        return "";
    }

}
