package com.tbyf.his.emport.service.impl;

import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.emport.core.EmportUtils;
import com.tbyf.his.emport.domain.vo.FieldInfoVO;
import com.tbyf.his.emport.entity.ExcelTemplate;
import com.tbyf.his.emport.entity.ExcelTemplateField;
import com.tbyf.his.emport.service.EmportService;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * @author lzz
 * @date 2023/2/3 9:34
 */

@Service
@Slf4j
@DataSource(DataSourceType.MASTER)
public class EmportServiceImpl implements EmportService {

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private TransactionTemplate transactionTemplate;


    @Override
    public void dynamicImportBySimple(ExcelTemplate template, List<FieldInfoVO> list, Workbook workbook) {
        final Sheet sheet = workbook.getSheetAt(0);
        //查询title行,一般是第一行
        int startIndex;
        int allRows = sheet.getLastRowNum() + 1;
        startIndex = IntStream.range(0, allRows)
                .filter(a -> EmportUtils.findText(sheet.getRow(a), list.get(0).getFieldTitle()))
                .findFirst().orElse(0);
        // 提取数据横坐标
        final Row titleRow = sheet.getRow(startIndex);
        Map<String, Integer> coordinate = new HashMap<>();
        titleRow.cellIterator().forEachRemaining(cell -> {
            final String value = EmportUtils.getValue(cell);
            if (StringUtils.isNotBlank(value)) {
                coordinate.put(value, cell.getColumnIndex());
            }
        });
        // 数据坐标清洗
        final List<FieldInfoVO> fieldInfoVOList = list.stream().peek(item -> {
                    if (StringUtils.isBlank(item.getFieldAddress())) {
                        if (StringUtils.isNotBlank(item.getFieldTitle())) {
                            final Integer integer = coordinate.get(item.getFieldTitle());
                            if (integer == null) {
                                log.info("字段坐标未找到:{}", item);
                            } else {
                                item.setHorizontal(integer);
                            }
                        }
                    } else {
                        item.setHorizontal(EmportUtils.getHorizontal(item.getFieldAddress()));
                    }
                    item.initFieldLocation();
                    item.initContentList();
                }).filter(item -> item.getHorizontal() != null)
                .collect(Collectors.toList());
        // 遍历获取数据,并进行规则判断
        List<List<String>> dataList = new ArrayList<>();
        for (int a = startIndex + 1; a < allRows; a++) {
            final Row row = sheet.getRow(a);
            List<String> tempList = new ArrayList<>();
            try {
                for (FieldInfoVO fieldInfoVO : fieldInfoVOList) {
                    String tempValue = EmportUtils.getValue(row.getCell(fieldInfoVO.getHorizontal()), fieldInfoVO.getDataType());
                    // 单位控制
                    /*if (StringUtils.isNotBlank(fieldInfoVO.getUnitTitle())) {
                        if (tempValue.endsWith(fieldInfoVO.getUnitTitle())) {
                            tempValue = tempValue.substring(0, tempValue.length() - fieldInfoVO.getUnitTitle().length()).trim();
                        }
                    }*/
                    // 校验器校验
                    EmportUtils.validate(fieldInfoVO.getRuleContentList(), tempValue);
                    tempList.add(tempValue);
                }
                dataList.add(tempList);
            } catch (Exception e) {
                log.error("数据清洗异常", e);
                throw e;
            }
        }
        // 批量添加数据
        final List<String> batchSql = EmportUtils.getBatchSql(template.getTablePartName(),
                template.getDataSourceId(),
                fieldInfoVOList,
                dataList,
                100);
        if (!CollectionUtils.isEmpty(batchSql)) {
            boolean autoCommit = true;
            Connection connection = null;
            try {
                DataSourceUtil.switchDs(template.getDataSourceId());
                TransactionSynchronizationManager.initSynchronization();
                connection = jdbcTemplate.getDataSource().getConnection();
                autoCommit = connection.getAutoCommit();
                connection.setAutoCommit(false);
                String[] sqls = new String[batchSql.size()];
                batchSql.toArray(sqls);
                jdbcTemplate.batchUpdate(sqls);
                connection.commit();
            } catch (Exception e) {
                try {
                    connection.rollback();
                } catch (SQLException ex) {
                    throw new RuntimeException(ex);
                }
                throw new RuntimeException(e);
            } finally {
                try {
                    connection.setAutoCommit(autoCommit);
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
                if (connection != null) {
                    try {
                        connection.close();
                    } catch (SQLException e) {
                        throw new RuntimeException(e);
                    }
                }
                DataSourceUtil.switchDefaultDs();
            }
        }
    }

    @Override
    public void dynamicImportByVertical(ExcelTemplate template, List<FieldInfoVO> list, Workbook workbook) {
        final Sheet sheet = workbook.getSheetAt(0);
        // 从第一个字段开始解析
        List<String> dataList = new ArrayList<>();
        list.forEach(fieldInfoVo -> {
            // 先判断是否有位置信息
            if (StringUtils.isBlank(fieldInfoVo.getFieldAddress())) {
                fieldInfoVo.setFieldAddress(EmportUtils.findAddress(sheet, fieldInfoVo));
            }
        });
        final List<FieldInfoVO> fieldList = list.stream()
                .filter(item -> StringUtils.isNotBlank(item.getFieldAddress()))
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(fieldList)) {
            fieldList.forEach(item -> {
                final String[] address = item.getFieldAddress().split(",");
                final Cell cell = sheet.getRow(Integer.parseInt(address[1])).getCell(Integer.parseInt(address[0]));
                String value = EmportUtils.getValue(cell, item.getDataType());
                // 处理这个值,以便兼容 5人/5
                if (StringUtils.isNotBlank(item.getUnitTitle())) {
                    if (value.endsWith(item.getUnitTitle())) {
                        value = value.substring(0, value.length() - item.getUnitTitle().length()).trim();
                    }
                }
                // 校验器校验
                EmportUtils.validate(item.getRuleContentList(), value);
                dataList.add(value);
            });
            // 拼接sql
            final String sql = EmportUtils.getInsertSql(template.getTablePartName(), template.getDataSourceId(),
                    fieldList, dataList);
            executeSql(template.getDataSourceId(), sql);
        }
    }

    @Override
    public List<ExcelTemplateField> initTemplateBySimple(ExcelTemplate template, Workbook workbook) {
        List<ExcelTemplateField> fieldList = new ArrayList<>();
        final Sheet sheet = workbook.getSheetAt(0);
        // 判断哪一行有数据有数据的就是标题
        final int i = EmportUtils.hasText(sheet);
        if (i >= 0) {
            for (Cell cell : sheet.getRow(i)) {
                ExcelTemplateField field = new ExcelTemplateField();
                final String value = EmportUtils.getValue(cell);
                if (StringUtils.isNotBlank(value)) {
                    field.setFieldTitle(value);
                    field.setTemplateId(template.getId())
                            .setSort((long) cell.getColumnIndex())
                            .setFieldLocation("1,1")
                            .setFieldAddress(cell.getColumnIndex() + "," + i);
                    field.initAdd();
                    fieldList.add(field);
                }
            }
        }
        return fieldList;
    }

    @Override
    public List<ExcelTemplateField> initTemplateByVertical(ExcelTemplate template, Workbook workbook) {
        List<ExcelTemplateField> fieldList = new ArrayList<>();
        final Sheet sheet = workbook.getSheetAt(0);
        int allRows = sheet.getLastRowNum() + 1;
        // 排除标题行
        root:
        for (int i = 0; i < allRows; i++) {
            ExcelTemplateField field = new ExcelTemplateField();
            final Row row = sheet.getRow(i);
            int size = 0;
            int x = 0;
            for (int y = 1; y < row.getLastCellNum(); y++) {
                Cell cell = row.getCell(y);
                // 排除标题以及特殊字符
                final String value = EmportUtils.getValue(cell);
                if (StringUtils.isNotBlank(value)) {
                    if (StringUtils.equalsAny(value, template.getSheetName(), template.getTemplateName(), "代码", "指标名称", "计量单位")) {
                        continue root;
                    }
                    if (StringUtils.isBlank(field.getTemplateFieldName())) {
                        field.setTemplateFieldName(value);
                        size++;
                        x = cell.getColumnIndex() + 1;
                    } else if (StringUtils.isBlank(field.getFieldTitle())) {
                        field.setFieldTitle(value);
                        size++;
                        x = cell.getColumnIndex() + 1;
                    } else {
                        field.setUnitTitle(value);
                        size++;
                        x = cell.getColumnIndex();
                        break;
                    }
                }
            }
            // 数据清洗
            // 如果一个数据都没有,就跳过
            // 如果只有一个数据,就把数据移动到标题上
            if (size == 0) {
                continue;
            } else if (size == 1) {
                field.setFieldTitle(field.getTemplateFieldName());
                field.setTemplateFieldName(null);
            }
            field.setTemplateId(template.getId())
                    .setSort((long) i)
                    .setFieldLocation("1,1")
                    .setFieldAddress(x + "," + i);
            field.initAdd();
            fieldList.add(field);
        }
        return fieldList;
    }

    public void executeSql(String dataSourceId, String sql) {
        try {
            DataSourceUtil.switchDs(dataSourceId);
            jdbcTemplate.execute(sql);
        } finally {
            DataSourceUtil.switchDefaultDs();
        }

    }


}
