package com.tbyf.his.emport.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.exception.base.BaseException;
import com.tbyf.his.emport.constant.EmportConstants;
import com.tbyf.his.emport.core.EmportConfig;
import com.tbyf.his.emport.core.RuleContent;
import com.tbyf.his.emport.core.RuleValidator;
import com.tbyf.his.emport.domain.param.AddExcelRuleParam;
import com.tbyf.his.emport.domain.param.QueryExcelRuleParam;
import com.tbyf.his.emport.domain.param.TestRuleParam;
import com.tbyf.his.emport.domain.param.UpdateExcelRuleParam;
import com.tbyf.his.emport.entity.ExcelFieldRule;
import com.tbyf.his.emport.entity.ExcelRule;
import com.tbyf.his.emport.mapper.ExcelRuleMapper;
import com.tbyf.his.emport.service.ExcelFieldRuleService;
import com.tbyf.his.emport.service.ExcelRuleService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/1/10 15:53
 */

@Slf4j
@Service
@DataSource(DataSourceType.MASTER)
public class ExcelRuleServiceImpl extends ServiceImpl<ExcelRuleMapper, ExcelRule> implements ExcelRuleService {

    @Autowired
    private ExcelRuleMapper excelRuleMapper;

    @Autowired
    private ExcelFieldRuleService excelFieldRuleService;

    @Autowired
    @Lazy
    private ExcelRuleService excelRuleService;

    @Override
    public TableDataInfo queryRule(QueryExcelRuleParam param) {
        final Page<ExcelRule> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<ExcelRule> queryWrapper = Wrappers.lambdaQuery(ExcelRule.class);
        queryWrapper.like(StringUtils.hasText(param.getRuleName()), ExcelRule::getRuleName, param.getRuleName())
                .eq(StringUtils.hasText(param.getRuleMode()), ExcelRule::getRuleMode, param.getRuleMode())
                .eq(StringUtils.hasText(param.getRuleType()), ExcelRule::getRuleType, param.getRuleType());
        final Page<ExcelRule> templatePage = excelRuleService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @Override
    public boolean deleteRule(Long ruleId) {
        // 需要判断这个规则别人是否使用过
        // 第一步是组合规则使用
        // 第二部是模板字段绑定规则使用
        final LambdaQueryWrapper<ExcelRule> wrapper1 = Wrappers.lambdaQuery(ExcelRule.class)
                .eq(ExcelRule::getRuleType, EmportConstants.RULE_COMBINE)
                .and(wrapper -> {
                    wrapper.likeRight(ExcelRule::getRuleContent, ruleId + ",")
                            .or()
                            .likeLeft(ExcelRule::getRuleContent, "," + ruleId);
                });
        final long count1 = excelRuleService.count(wrapper1);
        if (count1 > 0) {
            throw new BaseException("此规则已被其它组合规则使用");
        }
        final LambdaQueryWrapper<ExcelFieldRule> wrapper2 = Wrappers.lambdaQuery(ExcelFieldRule.class)
                .eq(ExcelFieldRule::getRuleId, ruleId);
        final long count2 = excelFieldRuleService.count(wrapper2);
        if (count2 > 0) {
            throw new BaseException("此规则已被模板字段绑定使用");
        }
        return excelRuleService.removeById(ruleId);
    }

    @Override
    public boolean addRule(AddExcelRuleParam param) {
        // 判断名称不相同
        final LambdaQueryWrapper<ExcelRule> queryWrapper = Wrappers.lambdaQuery(ExcelRule.class);
        queryWrapper.eq(ExcelRule::getRuleName, param.getRuleName());
        final long count = excelRuleService.count(queryWrapper);
        if (count > 0) {
            throw new BaseException("规则名重复");
        }
        ExcelRule rule = new ExcelRule();
        BeanUtils.copyProperties(param, rule);
        rule.initAdd();
        return excelRuleService.save(rule);
    }

    @Override
    public boolean updateRule(UpdateExcelRuleParam param) {
        // 判断名称不相同
        final LambdaQueryWrapper<ExcelRule> queryWrapper = Wrappers.lambdaQuery(ExcelRule.class);
        queryWrapper.eq(ExcelRule::getRuleName, param.getRuleName())
                .ne(ExcelRule::getId, param.getId());
        final long count = excelRuleService.count(queryWrapper);
        if (count > 0) {
            throw new BaseException("规则名重复");
        }
        ExcelRule rule = new ExcelRule();
        BeanUtils.copyProperties(param, rule);
        rule.initUpdate();
        return excelRuleService.updateById(rule);
    }

    @Override
    public List<ExcelRule> queryRuleDict() {
        final LambdaQueryWrapper<ExcelRule> queryWrapper = Wrappers.lambdaQuery(ExcelRule.class);
        queryWrapper.select(ExcelRule::getId, ExcelRule::getRuleName);
        return excelRuleService.list(queryWrapper);
    }

    @Override
    public AjaxResult testRule(TestRuleParam param) {
        if (!StringUtils.hasText(param.getValue())) {
            return AjaxResult.error("值不能为空");
        }
        RuleContent content = new RuleContent();
        content.setContent(param.getRuleContent());
        content.setType(param.getRuleType());
        content.setMode(param.getRuleMode());
        content.setValue(param.getValue());
        final RuleValidator validator = EmportConfig.getValidator(content.getMode());
        if (Objects.isNull(validator)) {
            return AjaxResult.error("校验服务加载异常,请联系管理员");
        }
        final boolean validate = validator.validate(content);
        return validate ? AjaxResult.success() : AjaxResult.error("数据校验不通过");
    }

    @Override
    public RuleContent getRuleContent(String id) {
        final ExcelRule rule = excelRuleService.getById(id);
        RuleContent content = new RuleContent();
        content.setContent(rule.getRuleContent());
        content.setId(id);
        content.setType(rule.getRuleType());
        content.setMode(rule.getRuleMode());
        content.setName(rule.getRuleName());
        return content;
    }

    @Override
    public List<String> getRuleNames(String ids) {
        final String[] idList = ids.split(",");
        final LambdaQueryWrapper<ExcelRule> wrapper = Wrappers.lambdaQuery(ExcelRule.class)
                .select(ExcelRule::getRuleName)
                .in(ExcelRule::getId, idList);
        return excelRuleService.list(wrapper).stream().map(ExcelRule::getRuleName).collect(Collectors.toList());
    }
}
