package com.tbyf.his.emport.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.exception.base.BaseException;
import com.tbyf.his.emport.domain.param.QueryTemplateParam;
import com.tbyf.his.emport.domain.param.UploadExcelParam;
import com.tbyf.his.emport.domain.vo.FieldInfoVO;
import com.tbyf.his.emport.entity.ExcelFieldRule;
import com.tbyf.his.emport.entity.ExcelTemplate;
import com.tbyf.his.emport.entity.ExcelTemplateField;
import com.tbyf.his.emport.mapper.ExcelTemplateMapper;
import com.tbyf.his.emport.service.EmportService;
import com.tbyf.his.emport.service.ExcelFieldRuleService;
import com.tbyf.his.emport.service.ExcelTemplateFieldService;
import com.tbyf.his.emport.service.ExcelTemplateService;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/1/10 15:53
 */

@Slf4j
@Service
@DataSource(DataSourceType.MASTER)
public class ExcelTemplateServiceImpl extends ServiceImpl<ExcelTemplateMapper, ExcelTemplate> implements ExcelTemplateService {

    @Autowired
    private ExcelTemplateMapper excelTemplateMapper;

    @Autowired
    private ExcelTemplateFieldService excelTemplateFieldService;

    @Autowired
    private ExcelFieldRuleService excelFieldRuleService;

    @Autowired
    @Lazy
    private ExcelTemplateService excelTemplateService;

    @Autowired
    private EmportService emportService;

    @Override
    public boolean addTemplate(ExcelTemplate template) {
        template.initAdd();
        // 判断相同名称模板
        final LambdaQueryWrapper<ExcelTemplate> queryWrapper = Wrappers.lambdaQuery(ExcelTemplate.class);
        queryWrapper.eq(ExcelTemplate::getTemplateName, template.getTemplateName());
        final long count = excelTemplateService.count(queryWrapper);
        if (count > 0) {
            throw new BaseException("已存在相同的模板名称");
        }
        return excelTemplateService.save(template);
    }

    @Override
    public boolean updateTemplate(ExcelTemplate template) {
        template.initUpdate();
        // 判断相同名称模板
        final LambdaQueryWrapper<ExcelTemplate> queryWrapper = Wrappers.lambdaQuery(ExcelTemplate.class);
        queryWrapper.eq(ExcelTemplate::getTemplateName, template.getTemplateName())
                .ne(ExcelTemplate::getId, template.getId());
        final long count = excelTemplateService.count(queryWrapper);
        if (count > 0) {
            throw new BaseException("已存在相同的模板名称");
        }
        return excelTemplateService.updateById(template);
    }

    @Override
    public TableDataInfo queryTemplate(QueryTemplateParam param) {
        final Page<ExcelTemplate> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<ExcelTemplate> queryWrapper = Wrappers.lambdaQuery(ExcelTemplate.class);
        queryWrapper.like(StringUtils.hasText(param.getKeyword()), ExcelTemplate::getTemplateName, param.getKeyword());
        final Page<ExcelTemplate> templatePage = excelTemplateService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteTemplate(Long templateId) {
        // 按照顺序删除,先删除字段规则绑定关系,再删除字段,再删除模板
        final ExcelTemplate template = excelTemplateService.getById(templateId);
        if (template == null) {
            return true;
        }
        final List<ExcelTemplateField> templateFieldList = excelTemplateFieldService.list(Wrappers.lambdaQuery(ExcelTemplateField.class).eq(ExcelTemplateField::getTemplateId, templateId));
        if (!CollectionUtils.isEmpty(templateFieldList)) {
            final List<Long> fieldIdList = templateFieldList.stream().map(ExcelTemplateField::getId).collect(Collectors.toList());
            excelFieldRuleService.remove(Wrappers.lambdaQuery(ExcelFieldRule.class).in(ExcelFieldRule::getFieldId, fieldIdList));
            excelTemplateFieldService.removeByIds(fieldIdList);
        }
        return excelTemplateService.removeById(templateId);
    }

    @Override
    public boolean importExcel(UploadExcelParam param) {
        try (InputStream is = param.getFile().getInputStream(); Workbook workbook = WorkbookFactory.create(is)) {
            final Sheet sheet = workbook.getSheetAt(0);
            // 查询模板信息,与相对应的所有字段信息
            final ExcelTemplate template = excelTemplateService.getById(param.getTemplateId());
            if (template == null) {
                throw new BaseException("模板ID不存在");
            }
            // 查询字段与模板对应的多对多列表
            List<FieldInfoVO> list = excelTemplateMapper.getFieldInfoList(param.getTemplateId());
            if (CollectionUtils.isEmpty(list)) {
                throw new BaseException("该模板未配置字段");
            }
            if (com.tbyf.his.common.utils.StringUtils.equals(template.getTemplateType(), "1")) {
                emportService.dynamicImportBySimple(template, list, workbook);
            } else if (com.tbyf.his.common.utils.StringUtils.equals(template.getTemplateType(), "2")) {
                emportService.dynamicImportByVertical(template, list, workbook);
            } else {
                throw new BaseException("未知的文档类型");
            }
            return true;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void initTemplate(UploadExcelParam param) {
        try (InputStream is = param.getFile().getInputStream(); Workbook workbook = WorkbookFactory.create(is)) {
            final ExcelTemplate template = excelTemplateService.getById(param.getTemplateId());
            if (template == null) {
                throw new BaseException("未查询到对应的模板");
            }
            List<ExcelTemplateField> fieldList;
            if (com.tbyf.his.common.utils.StringUtils.equals("1", template.getTemplateType())) {
                fieldList = emportService.initTemplateBySimple(template, workbook);
            } else {
                fieldList = emportService.initTemplateByVertical(template, workbook);
            }
            if (!CollectionUtils.isEmpty(fieldList)) {
                excelTemplateService.initTemplateField(template.getId(), fieldList);
            }

            final Sheet sheet = workbook.getSheetAt(0);
            int rows = sheet.getLastRowNum() + 1;
            int x = 0;
            // 查询对应的数据的列数
            int code = 0, title = 0, unit = 0;
            for1:
            for (; x < rows; x++) {
                // 查找从第几行开始
                final Row row = sheet.getRow(x);
                int index = 0;
                final Iterator<Cell> cellIterator = row.cellIterator();
                while (cellIterator.hasNext()) {
                    final Cell cell = cellIterator.next();
                    final String cellValue = getStringCellValue(cell);
                    if (index == 0 && com.tbyf.his.common.utils.StringUtils.equals(cellValue, "代码")) {
                        index++;
                        code = cell.getColumnIndex();
                    } else if (index == 1 && com.tbyf.his.common.utils.StringUtils.equals(cellValue, "指标名称")) {
                        index++;
                        title = cell.getColumnIndex();
                    } else if (index == 2 && com.tbyf.his.common.utils.StringUtils.equals(cellValue, "计量单位")) {
                        unit = cell.getColumnIndex();
                        break for1;
                    }
                }
            }
            for (int i = x + 1; i < rows; i++) {
                // 查找从第几行开始
                final Row row = sheet.getRow(i);
                final String fieldName = getStringCellValue(row.getCell(code));
                if (StringUtils.hasText(fieldName)) {
                    ExcelTemplateField field = new ExcelTemplateField();
                    field.setTemplateId(Long.valueOf(param.getTemplateId()));
                    field.setTemplateFieldName(fieldName);
                    field.setSort((long) i);
                    field.setFieldTitle(getStringCellValue(row.getCell(title)));
                    final String unitTitle = getStringCellValue(row.getCell(unit));
                    field.setUnitTitle(com.tbyf.his.common.utils.StringUtils.equals(unitTitle, "-") ? "" : unitTitle);
                    field.initAdd();
                    fieldList.add(field);
                }
            }
            excelTemplateService.initTemplateField(Long.valueOf(param.getTemplateId()), fieldList);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void initTemplateField(Long templateId, List<ExcelTemplateField> list) {
        final List<ExcelTemplateField> templateFields = excelTemplateFieldService.list(
                Wrappers.lambdaQuery(ExcelTemplateField.class)
                        .eq(ExcelTemplateField::getTemplateId, templateId)
        );
        templateFields.forEach(field -> {
            excelTemplateFieldService.deleteTemplateField(field.getId());
        });
        excelTemplateFieldService.saveBatch(list);
    }

    @Override
    public void downloadTemplate(Long templateId, HttpServletResponse response) {
        final ExcelTemplate template = excelTemplateService.getById(templateId);
        if (template == null) {
            throw new BaseException("未查询到对应模板");
        }
        final List<ExcelTemplateField> fieldList = excelTemplateFieldService.list(
                Wrappers.lambdaQuery(ExcelTemplateField.class)
                        .eq(ExcelTemplateField::getTemplateId, template.getId())
                        .orderByAsc(ExcelTemplateField::getSort)
        );
        try (Workbook workbook = WorkbookFactory.create(true)) {
            final Sheet sheet = workbook.createSheet(template.getRemark());
            int index = 0;
            final Row titleRow = sheet.createRow(index++);
            final Cell titleCell = titleRow.createCell(1);
            titleCell.setCellValue(template.getRemark());
            final CellStyle titleStyle = workbook.createCellStyle();
            titleStyle.setAlignment(HorizontalAlignment.CENTER);
            titleStyle.setVerticalAlignment(VerticalAlignment.CENTER);
            titleCell.setCellStyle(titleStyle);
            // 合并标题单元格
            sheet.addMergedRegion(new CellRangeAddress(0, 0, 1, 4));
            // 样式
            sheet.setColumnWidth(0, 0);
            sheet.setColumnWidth(1, 2867);
            sheet.setColumnWidth(2, 6400);
            sheet.setColumnWidth(3, 7936);
            sheet.setColumnWidth(4, 2739);
            sheet.setDefaultRowHeight((short) 360);

            workbook.write(response.getOutputStream());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }

    public String getStringCellValue(Cell cell) {
        final CellType cellType = cell.getCellType();
        switch (cellType) {
            case STRING:
                return cell.getStringCellValue().trim();
            case BOOLEAN:
                return String.valueOf(cell.getBooleanCellValue());
            case NUMERIC:
                return ((XSSFCell) cell).getCTCell().getV();
            default:
                return "";
        }
    }

}
