package com.tbyf.his.web.controller.apiconvert;

import com.tbyf.his.apiconvert.domain.FieldMap;
import com.tbyf.his.apiconvert.domain.model.SourceAndTarget;
import com.tbyf.his.apiconvert.service.IFieldMappingService;
import com.tbyf.his.common.annotation.Log;
import com.tbyf.his.common.constant.UserConstants;
import com.tbyf.his.common.core.controller.BaseController;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.domain.entity.FieldMapping;
import com.tbyf.his.common.enums.BusinessType;
import com.tbyf.his.common.utils.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Iterator;
import java.util.List;

/**
 * 字段映射
 *
 * @author guopx
 */
@RestController
@RequestMapping("/system/field")
public class FieldMappingController extends BaseController {
    @Autowired
    private IFieldMappingService FieldService;

    /**
     * 获取字段映射列表
     */
    @GetMapping("/list")
    public AjaxResult list(FieldMapping Field) {
        List<FieldMapping> Fields = FieldService.selectFieldList(Field);
        return AjaxResult.success(Fields);
    }

    @GetMapping("/selectConnectTo")
    public AjaxResult selectConnectTo(FieldMapping Field) {
        List<FieldMapping> Fields = FieldService.selectConnectTo(Field);
        return AjaxResult.success(Fields);
    }

    /**
     * 查询字段映射列表（排除节点）
     */
    @GetMapping("/list/exclude/{FieldId}")
    public AjaxResult excludeChild(@PathVariable(value = "FieldId", required = false) Long FieldId) {
        List<FieldMapping> Fields = FieldService.selectFieldList(new FieldMapping());
        Iterator<FieldMapping> it = Fields.iterator();
        while (it.hasNext()) {
            FieldMapping d = (FieldMapping) it.next();
            if (d.getFieldId().intValue() == FieldId
                    || ArrayUtils.contains(StringUtils.split(d.getAncestors(), ","), FieldId + "")) {
                it.remove();
            }
        }
        return AjaxResult.success(Fields);
    }

    /**
     * 根据字段映射编号获取详细信息
     */
    @GetMapping(value = "/{FieldId}")
    public AjaxResult getInfo(@PathVariable Long FieldId) {
        return AjaxResult.success(FieldService.selectFieldById(FieldId));
    }

    /**
     * 获取字段映射下拉树列表
     */
    @GetMapping("/treeselect")
    public AjaxResult treeselect(FieldMapping Field) {
        List<FieldMapping> Fields = FieldService.selectFieldList(Field);
        return AjaxResult.success(FieldService.buildFieldTreeSelect(Fields));
    }

    /**
     * 新增字段映射
     */
    @Log(title = "字段映射管理", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@Validated @RequestBody FieldMapping Field) {
        if (UserConstants.NOT_UNIQUE.equals(FieldService.checkFieldNameUnique(Field))) {
            return AjaxResult.error("新增字段映射'" + Field.getFieldName() + "'失败，字段映射名称已存在");
        }
        Field.setCreateBy(getUsername());
        return toAjax(FieldService.insertField(Field));
    }

    /**
     * 修改字段映射
     */
    @Log(title = "字段映射管理", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@Validated @RequestBody FieldMapping Field) {
        Long FieldId = Field.getFieldId();
        if (UserConstants.NOT_UNIQUE.equals(FieldService.checkFieldNameUnique(Field))) {
            return AjaxResult.error("修改字段映射'" + Field.getFieldName() + "'失败，字段映射名称已存在");
        } else if (Field.getParentId().equals(FieldId)) {
            return AjaxResult.error("修改字段映射'" + Field.getFieldName() + "'失败，上级字段映射不能是自己");
        }
        Field.setUpdateBy(getUsername());
        return toAjax(FieldService.updateField(Field));
    }

    /**
     * 删除字段映射
     */
    @Log(title = "字段映射管理", businessType = BusinessType.DELETE)
    @DeleteMapping("/{FieldId}")
    public AjaxResult remove(@PathVariable Long FieldId) {
        if (FieldService.hasChildByFieldId(FieldId)) {
            return AjaxResult.error("存在下级字段映射,不允许删除");
        }
        return toAjax(FieldService.deleteFieldById(FieldId));
    }


}
