package com.tbyf.his.web.dataImport.core;

import com.tbyf.his.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author lzz
 * @date 2023/2/9 9:22
 */
@Configuration
@Slf4j
public class DiConfig {

    @Autowired
    private ApplicationContext applicationContext;

    public static final Map<String, RuleValidator> VALIDATOR_MAP = new ConcurrentHashMap<>(8);

    public static final List<String> WORD = Arrays.asList(
            "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z"
    );

    @PostConstruct
    public void init() {
        final Map<String, RuleValidator> validatorMap = applicationContext.getBeansOfType(RuleValidator.class);
        VALIDATOR_MAP.putAll(validatorMap);
    }

    /**
     * 获取校验器
     *
     * @param mode
     * @return
     */
    public static RuleValidator getValidator(String mode) {
        return VALIDATOR_MAP.get(DiConstants.SERVICE_PREFIX + mode);
    }

    public static int getIndex(String word) {
        for (int i = 0; i < WORD.size(); i++) {
            if (StringUtils.equals(word, WORD.get(i))) {
                return i;
            }
        }
        return -1;
    }

    public static String getWord(int index) {
        return WORD.get(index);
    }

    /**
     * 获取单元格的值
     *
     * @param cell
     * @return
     */
    public static String getValue(Cell cell) {
        if (cell == null || "".equals(cell.toString().trim())) {
            return "";
        }
        String cellString = "";
        final CellType type = cell.getCellType();
        switch (type) {
            case STRING:
                cellString = trim(cell.getRichStringCellValue().getString());
                break;
            case BOOLEAN:
                cellString = String.valueOf(cell.getBooleanCellValue());
                break;
            case FORMULA:
                final FormulaEvaluator evaluator = cell.getSheet().getWorkbook().getCreationHelper().createFormulaEvaluator();
                cellString = evaluator.evaluate(cell).getStringValue().trim();
                break;
            case NUMERIC:
                final short format = cell.getCellStyle().getDataFormat();
                if (DateUtil.isCellDateFormatted(cell)) {
                    // 日期格式
                    SimpleDateFormat sdf = null;
                    if (format == 20 || format == 32) {
                        sdf = new SimpleDateFormat("HH:mm");
                        cellString = sdf.format(cell.getDateCellValue());
                    } else if (format == 14 || format == 31 || format == 57 || format == 58) {
                        // 处理自定义日期格式：m月d日(通过判断单元格的格式id解决，id的值是58)
                        sdf = new SimpleDateFormat("yyyy-MM-dd");
                        double value = cell.getNumericCellValue();
                        Date date = org.apache.poi.ss.usermodel.DateUtil
                                .getJavaDate(value);
                        cellString = sdf.format(date);
                    } else {// 日期
                        sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        cellString = sdf.format(cell.getDateCellValue());
                    }
                } else {
                    cellString = NumberToTextConverter.toText(cell.getNumericCellValue());
                }
                break;
            default:
                break;
        }
        return cellString;
    }

    public static String trim(String text) {
        return text.replaceAll("(\\u00A0+| )", "").trim();
    }

}
