package com.tbyf.his.job.task;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tbyf.his.apiconvert.domain.ApiconvertBaseInfoModel;
import com.tbyf.his.apiconvert.service.IApiconvertBaseinfoService;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.framework.system.service.ISysConfigService;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

@Component("healthCardTask")
@Slf4j
public class HealthCardTask {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private ISysConfigService iSysConfigService;

    @Autowired
    IApiconvertBaseinfoService apiconvertBaseinfoService;

    @Autowired
    private TaskExecutor taskExecutor;

    public static String SOAP_TEMPLATE = "<soap:Envelope xmlns:soap=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:ser=\"http://service.webservice.platform.his.tbyf.com/\">\n" + "   <soap:Header/>\n" + "   <soap:Body>\n" + "      <ser:indexRegister>\n" + "         <xml>\n" + "         <![CDATA[\n" + "         <Program>\n" + "    <parameters>\n" + "        <sourceid>WXGZH</sourceid>\n" + "        <synccode>N00022</synccode>\n" + "        <sourcetype>I</sourcetype>\n" + "        <syncLogCode>b5f9feaafb114bec8afa07da85cd2110</syncLogCode>\n" + "        <datasetcode>D02919</datasetcode>\n" + "    </parameters>\n" + "    <D02919>{}</D02919>\n" + "</Program>\n" + "         ]]>\n" + "         </xml>\n" + "      </ser:indexRegister>\n" + "   </soap:Body>\n" + "</soap:Envelope>";

    private static final String SQL = "SELECT\n" + "\t* \n" + "FROM\n" + "\t(\n" + "\tSELECT\n" + "\t\tA.*,\n" + "\t\tROWNUM RN \n" + "\tFROM\n" + "\t\t(\n" + "\t\tSELECT\n" + "\t\t\tJGDM AS orgCode,\n" + "\t\t\tID AS sysRECId,\n" + "\t\t\tBRLX AS categoryCode,\n" + "\t\t\tSFZJLB AS idTypeCode,\n" + "\t\t\tSFZH AS idNo,\n" + "\t\t\tBRXM AS name,\n" + "\t\t\tCSRQ AS dob,\n" + "\t\t\tBRXB AS genderName,\n" + "\t\t\tXX AS rhBloodTypeCode,\n" + "\t\t\tHYZK AS maritalStatusName,\n" + "\t\t\tGJ AS citizenshipCode,\n" + "\t\t\tMZ AS ethnicGroupCode,\n" + "\t\t\tBRZY AS occupationCode,\n" + "\t\t\tLXDH AS phoneNo \n" + "\t\tFROM\n" + "\t\t\tWEB_SYS_USER \n" + "\t\tORDER BY\n" + "\t\t\tJDSJ DESC \n" + "\t\t) A \n" + "\tWHERE\n" + "\t\tROWNUM <= {}\n" + "\t) \n" + "WHERE\n" + "\tRN >={}";


    /**
     * 存储运行情况,key为orgCode value为数量
     */
    public static final Map<String, AtomicLong> INDEX_MAP = new HashMap<>();

    public static final Map<String, AtomicBoolean> EXECUTE_MAP = new HashMap<>();

    public void healthRegister() {
        String empiUrl = iSysConfigService.selectConfigByKey("EMPI_URL");
        restTemplate.getForEntity(empiUrl + "/open/api/handlerRequestNoAuth/repeatRegister", null);
    }

    /**
     * 索引重置
     *
     * @param num 如果是-1,就查询当前数量重置  如果是-2,就只展示信息
     */
    public void reset(String orgCode, String num) {
        if ("-2".equals(num)) {
            log.info("======================当前各任务执行情况为:[{}]=======================", JSON.toJSONString(INDEX_MAP));
            log.info("======================当前各任务调度情况为:[{}]=======================", JSON.toJSONString(EXECUTE_MAP));
            return;
        }
        long count = Long.parseLong(num);
        if ("-1".equals(num)) {
            count = getCount(orgCode);
        }
        AtomicLong atomicLong = INDEX_MAP.get(orgCode);
        if (atomicLong == null) {
            atomicLong = new AtomicLong(count);
            INDEX_MAP.put(orgCode, atomicLong);
        } else {
            atomicLong.set(count);
        }
        log.info("======================主索引注册调度重置=======================");
        log.info("======================当前各任务执行情况为:[{}]=======================", JSON.toJSONString(INDEX_MAP));
    }

    public long getCount(String orgCode) {
        String empiUrl = iSysConfigService.selectConfigByKey("EMPI_URL") + "/open/api/handlerRequestNoAuth/countEmpi/" + orgCode;
        final AjaxResult result = restTemplate.getForObject(empiUrl, AjaxResult.class);
        final Object o = result.get(AjaxResult.DATA_TAG);
        if (o == null) {
            return 0L;
        }
        try {
            return Long.parseLong(String.valueOf(o));
        } catch (Exception ignore) {
            return 0L;
        }
    }

    /**
     * 当前是否可执行
     *
     * @param orgCode
     * @return
     */
    public boolean isExecute(String orgCode) {
        AtomicBoolean aBoolean = EXECUTE_MAP.get(orgCode);
        if (aBoolean != null) {
            return aBoolean.get();
        } else {
            aBoolean = new AtomicBoolean(true);
            EXECUTE_MAP.put(orgCode, aBoolean);
            return aBoolean.get();
        }
    }

    /**
     * 修改执行状态
     *
     * @param orgCode
     * @param init
     */
    public void resetExecute(String orgCode, boolean init) {
        AtomicBoolean aBoolean = EXECUTE_MAP.get(orgCode);
        if (aBoolean != null) {
            aBoolean.set(init);
        } else {
            aBoolean = new AtomicBoolean(init);
            EXECUTE_MAP.put(orgCode, aBoolean);
        }
    }

    public long getIndex(String orgCode) {
        AtomicLong atomicLong = INDEX_MAP.get(orgCode);
        if (atomicLong == null) {
            atomicLong = new AtomicLong(getCount(orgCode));
            INDEX_MAP.put(orgCode, atomicLong);
        }
        return atomicLong.get();
    }

    public void addIndex(String orgCode, long num) {
        AtomicLong atomicLong = INDEX_MAP.get(orgCode);
        if (atomicLong == null) {
            atomicLong = new AtomicLong(num);
            INDEX_MAP.put(orgCode, atomicLong);
        }
        atomicLong.addAndGet(num);
    }


    /**
     * @param orgCode     机构ID
     * @param size        单次查询条数
     * @param threadCount 开启线程数
     */
    public void batchRegister(String apiId, String orgCode, String size, String threadCount) {
        if (isExecute(orgCode)) {
            log.info("主索引注册任务:机构代码:[{}],本次查询条数:[{}],开启线程数:[{}]", orgCode, size, threadCount);
            resetExecute(orgCode, false);
            List<JSONObject> personList;
            try {
                long pageStart = getIndex(orgCode);
                final long start = pageStart + 1;
                final long end = pageStart + Long.parseLong(size);
                log.info("========================主索引注册任务,orgCode:[{}]:开始:[{}],结束:[{}]=======================", orgCode, start, end);
                final ApiconvertBaseInfoModel baseInfoModel = new ApiconvertBaseInfoModel();
                baseInfoModel.setApiId(Long.parseLong(apiId));
                final JSONObject param = new JSONObject();
                param.put("start", start);
                param.put("end", end);
                baseInfoModel.setOrgCode(Long.parseLong(orgCode));
                baseInfoModel.setParam(param.toJSONString());
                personList = (List<JSONObject>) apiconvertBaseinfoService.handlerRequest(baseInfoModel);
            } catch (Exception e) {
                log.error("=======================接口转换sql查询异常======================", e);
                resetExecute(orgCode, true);
                return;
            }
            if (CollectionUtils.isNotEmpty(personList)) {
                final long startTime = System.currentTimeMillis();
                final String empiUrl = iSysConfigService.selectConfigByKey("EMPI_URL") + "/open/api/handlerRequestNoAuth/webservice/empi?wsdl";
                final int count = personList.size();
                final Queue<JSONObject> personQueue = new ArrayBlockingQueue<>(count);
                personQueue.addAll(personList);
                // 动态开线程数量
                int threadNum = Integer.parseInt(threadCount);
                CompletableFuture<Void>[] futures = new CompletableFuture[threadNum];
                for (int x = 0; x < threadNum; x++) {
                    futures[x] = CompletableFuture.runAsync(() -> {
                        try {
                            JSONObject object = personQueue.poll();
                            while (object != null) {
                                final EMPIPerson person = object.toJavaObject(EMPIPerson.class);
                                if (StringUtils.isNotEmpty(person.getDob())) {
                                    person.setDob(person.getDob().replace("/", "-"));
                                }
                                String xml = objectToXML(person);
                                doSoap(empiUrl, StrFormatter.format(SOAP_TEMPLATE, xml));
                                object = personQueue.poll();
                            }
                        } catch (Exception e) {
                            log.error("执行任务异常", e);
                        }
                    }, taskExecutor);
                }
                CompletableFuture.allOf(futures).whenComplete((s, ex) -> {
                    log.info("========================主索引注册任务,orgCode:[{}]:执行时间:[{}],执行数量:[{}]=======================", orgCode, System.currentTimeMillis() - startTime, count);
                    addIndex(orgCode, count);
                    resetExecute(orgCode, true);
                });
            } else {
                log.info("========================主索引注册任务,orgCode:[{}],档案已注册完毕,可适当调低线程,当前总数量:[{}]=======================", orgCode, getIndex(orgCode));
                resetExecute(orgCode, true);
            }
        } else {
            log.info("========================主索引注册任务,orgCode:[{}],上次任务还未执行完毕,请等待,当前总数量:[{}]=======================", orgCode, getIndex(orgCode));
        }
    }

    private static final Marshaller.Listener marListener = new Marshaller.Listener() {
        @Override
        public void beforeMarshal(Object source) {
            Field[] fields = source.getClass().getDeclaredFields();
            for (Field f : fields) {
                f.setAccessible(true);
                try {
                    //对象为空且类型为String时候设置空值
                    if (f.getType() == String.class && f.get(source) == null) {
                        f.set(source, "");
                    }
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        }
    };

    public static String objectToXML(Object object) throws JAXBException {
        JAXBContext context = JAXBContext.newInstance(object.getClass());
        Marshaller m = context.createMarshaller();
        // 设置格式化输出
        m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
        // 决定时候需要xml文件头
        m.setProperty(Marshaller.JAXB_FRAGMENT, true);
        m.setListener(marListener);
        Writer w = new StringWriter();
        m.marshal(object, w);
        return w.toString();
    }

    @XmlRootElement(name = "row")
    @XmlAccessorType(value = XmlAccessType.FIELD)
    @Data
    public static class EMPIPerson {

        @XmlElement(name = "DE01.00.012.20")
        private String sysCode = "99";

        @XmlElement(name = "DE08.10.052.00")
        private String orgCode;

        @XmlElement(name = "DE02.01.062.47")
        private String sysRECId;

        @XmlElement(name = "DE01.00.002.14")
        private String categoryCode;

        @XmlElement(name = "DE02.01.031.00")
        private String idTypeCode;

        @XmlElement(name = "DE02.01.030.00")
        private String idNo;

        @XmlElement(name = "DE02.01.039.65")
        private String name;

        @XmlElement(name = "DE02.01.005.01")
        private String dob;

        @XmlElement(name = "DE02.01.040.03")
        private String genderName;

        @XmlElement(name = "DE02.01.018.01")
        private String maritalStatusName;

        @XmlElement(name = "DE02.01.025.00")
        private String ethnicGroupCode;

        @XmlElement(name = "DE02.01.052.10")
        private String occupationCode;

        @XmlElement(name = "DE02.01.010.00")
        private String phoneNo;

    }

    public static String doSoap(String postUrl, String soapXml) {
        String retStr = "";
        // 创建HttpClientBuilder
        HttpClientBuilder httpClientBuilder = HttpClientBuilder.create();
        // HttpClient
        CloseableHttpClient closeableHttpClient = httpClientBuilder.build();
        HttpPost httpPost = new HttpPost(postUrl);
        // 设置请求和传输超时时间
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(20000).setConnectTimeout(20000).build();
        httpPost.setConfig(requestConfig);
        try {
            httpPost.setHeader("Content-Type", "text/xml; charset=utf-8");
            StringEntity data = new StringEntity(soapXml, StandardCharsets.UTF_8);
            httpPost.setEntity(data);
            CloseableHttpResponse response = closeableHttpClient.execute(httpPost);
            HttpEntity httpEntity = response.getEntity();
            if (httpEntity != null) {
                // 打印响应内容
                retStr = EntityUtils.toString(httpEntity, "UTF-8");
            }
        } catch (Exception e) {
            log.error("调用ws异常", e);
        } finally {
            try {
                // 释放资源
                closeableHttpClient.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return retStr;
    }

    public static void main(String[] args) {
//        String url = "http://192.168.20.86:28081/open/api/handlerRequestNoAuth/webservice/empi?wsdl";
//        String xml = "<soap:Envelope xmlns:soap=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:ser=\"http://service.webservice.platform.his.tbyf.com/\">\n" + "   <soap:Header/>\n" + "   <soap:Body>\n" + "      <ser:indexRegister>\n" + "         <!--Optional:-->\n" + "         <xml>\n" + "         \n" + "         123\n" + "         </xml>\n" + "      </ser:indexRegister>\n" + "   </soap:Body>\n" + "</soap:Envelope>";
//        doSoap(url, xml);
    }


}
