package com.tbyf.his.web.controller.system;

import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.controller.BaseController;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.domain.model.MyKeyValue;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.utils.poi.ExcelUtil;
import com.tbyf.his.system.domain.SysDataTableVo;
import com.tbyf.his.system.domain.SysDatasource;
import com.tbyf.his.system.service.ISysDatasourceService;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 数据源配置Controller
 *
 * @author guopx
 * @date 2022-03-01
 */
@RestController
@RequestMapping("/system/datasource")
public class SysDatasourceController extends BaseController {
    @Autowired
    private ISysDatasourceService sysDatasourceService;

    /**
     * 查询数据源配置列表
     */
    @IgnoreWebSecurity
    @GetMapping("/list")
    public TableDataInfo list(SysDatasource sysDatasource) {
        startPage();
        List<SysDatasource> list = sysDatasourceService.selectSysDatasourceList(sysDatasource);
        return getDataTable(list);
    }


    @IgnoreWebSecurity
    /**
     * 查询具有适配器标识的数据源树列表
     */
    @GetMapping("/select")
    public AjaxResult select() {
        List<MyKeyValue> dataSource = sysDatasourceService.selectSysDatasource();
        return AjaxResult.success(dataSource);
    }

    /**
     * 导出数据源配置列表
     */
    @IgnoreWebSecurity
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysDatasource sysDatasource) {
        List<SysDatasource> list = sysDatasourceService.selectSysDatasourceList(sysDatasource);
        ExcelUtil<SysDatasource> util = new ExcelUtil<SysDatasource>(SysDatasource.class);
        util.exportExcel(response, list, "数据源配置数据");
    }

    /**
     * 获取数据源配置详细信息
     */
    @IgnoreWebSecurity
    @GetMapping(value = "/{datasourceId}")
    public AjaxResult getInfo(@PathVariable("datasourceId") Long datasourceId) {
        return AjaxResult.success(sysDatasourceService.selectSysDatasourceByDatasourceId(datasourceId));
    }

    /**
     * 新增数据源配置
     */
    @IgnoreWebSecurity
    @PostMapping
    public AjaxResult add(@RequestBody SysDatasource sysDatasource) {
        return toAjax(sysDatasourceService.insertSysDatasource(sysDatasource));
    }

    /**
     * 修改数据源配置
     */
    @IgnoreWebSecurity
    @PutMapping
    public AjaxResult edit(@RequestBody SysDatasource sysDatasource) {
        return toAjax(sysDatasourceService.updateSysDatasource(sysDatasource));
    }

    /**
     * 删除数据源配置
     */
    @IgnoreWebSecurity
    @DeleteMapping("/{datasourceIds}")
    public AjaxResult remove(@PathVariable Long[] datasourceIds) {
        return toAjax(sysDatasourceService.deleteSysDatasourceByDatasourceIds(datasourceIds));
    }

    @GetMapping(value = "/getTablesBySourceId/{id}")
    @IgnoreWebSecurity
    public AjaxResult getTablesBySourceId(@PathVariable("id") Long id) {
        return AjaxResult.success(sysDatasourceService.getTablesBySourceId(id));
    }

    /**
     * 测试连通性
     */
    @GetMapping(value = "/getTestConnectBySource")
    @IgnoreWebSecurity
    public AjaxResult testConnect(SysDatasource sysDatasource) {
        sysDatasourceService.testConnectBySourceId(sysDatasource);
        return AjaxResult.success();
    }

    /**
     * 根据数据源和表名获取表分页
     *
     * @param sysDataTableVo
     * @return
     */
    @GetMapping(value = "/getTablesBySourceIdAndTableName")
    @IgnoreWebSecurity
    public AjaxResult getTablesBySourceIdAndTableName(SysDataTableVo sysDataTableVo) {
        return AjaxResult.success(sysDatasourceService.getTablesBySourceIdAndTableName(sysDataTableVo));
    }

    @ApiOperation("getColumnsByTableName")
    @GetMapping(value = "/getColumnsByTableName/{id}/{tableName}")
    @IgnoreWebSecurity
    public AjaxResult getColumnsByTableName(@PathVariable("id") Long id, @PathVariable("tableName") String tableName) {
        return AjaxResult.success(sysDatasourceService.getColumnsByTableName(id, tableName));
    }

}
