package com.tbyf.his.web.dataImport.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.poi.ExcelUtil;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.core.RuleVO;
import com.tbyf.his.web.dataImport.core.RuleValidator;
import com.tbyf.his.web.dataImport.domain.vo.ExcelVO;
import com.tbyf.his.web.dataImport.domain.vo.VerifyVO;
import com.tbyf.his.web.dataImport.entity.DataImportTemplate;
import com.tbyf.his.web.dataImport.entity.ExcelData;
import com.tbyf.his.web.dataImport.mapper.ExcelDataMapper;
import com.tbyf.his.web.dataImport.service.DataImportTemplateService;
import com.tbyf.his.web.dataImport.service.ExcelDataService;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 11:24
 */
@Slf4j
@Service
@DataSource(DataSourceType.MASTER)
public class ExcelDataServiceImpl extends ServiceImpl<ExcelDataMapper, ExcelData> implements ExcelDataService {

    @Autowired
    private DataImportTemplateService dataImportTemplateService;

    @Autowired
    private JdbcTemplate jdbcTemplate;


    @Override
    public void analyzeExport(ExcelData excelData, HttpServletResponse response) {
        final DataImportTemplate template = dataImportTemplateService.getById(excelData.getTemplateId());
        List<VerifyVO> verifyList = dataImportTemplateService.getVerify(template.getId());
        try (InputStream is = new ByteArrayInputStream(excelData.getFile()); Workbook workbook = WorkbookFactory.create(is)) {
            final Sheet sheet = workbook.getSheetAt(0);
            //final int rows = sheet.getLastRowNum() + 1;
            final Optional<VerifyVO> first = verifyList.stream()
                    .filter(item -> StringUtils.isNotBlank(item.getCoordinate()))
                    .findFirst();
            // 数据长度/条数
            final int length = DiConfig.getLength(first.get(), sheet);
            // 第一步解析数据到values
            for (int a = 0; a < verifyList.size(); a++) {
                final VerifyVO vo = verifyList.get(a);
                if (StringUtils.isBlank(vo.getCoordinate())) {
                    final String s = DiConfig.getCoordinate(vo, sheet);
                    if (StringUtils.isBlank(s)) {
                        vo.addError(new ExcelVO("没有符合代码与指标名称的字段行", ""));
                        continue;
                    }
                    vo.setCoordinate(s);
                }
                final String[] coords = vo.getCoordinate().split(",");
                int columnStart = DiConfig.getIndex(coords[0]);
                int rowStart = Integer.parseInt(coords[1]);
                final Row row = sheet.getRow(rowStart);
                for (int b = columnStart; b < columnStart + length; b++) {
                    try {
                        final Cell cell = row.getCell(b, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
                        vo.addValue(DiConfig.getValue(cell));
                    } catch (Exception ignore) {
                    }
                }
            }
            // 获取每一步的核心机构等信息
            ExcelVO[] excels = new ExcelVO[length];
            for (int e = 0; e < length; e++) {
                excels[e] = new ExcelVO();
                excels[e].setOrgType(template.getOrgName());
                for (VerifyVO v : verifyList) {
                    if (StringUtils.equals("ORG_NAME", v.getField())) {
                        excels[e].setOrgName(v.getValues().get(e));
                        continue;
                    }
                    if (StringUtils.equals("ORG_CODE", v.getField())) {
                        excels[e].setOrgCode(v.getValues().get(e));
                        continue;
                    }
                    if (StringUtils.equals("UNIFIED_CODE", v.getField())) {
                        excels[e].setUnifiedCode(v.getValues().get(e));
                        continue;
                    }
                }
            }
            // 第二步进行数据校验
            verifyList.stream()
                    .filter(item -> !CollectionUtils.isEmpty(item.getValues()) && !CollectionUtils.isEmpty(item.getRules()))
                    .forEach(item -> {
                        for (int i = 0; i < item.getValues().size(); i++) {
                            for (int j = 0; j < item.getRules().size(); j++) {
                                final RuleVO vo = item.getRules().get(j);
                                vo.setValue(item.getValues().get(i));
                                // 置空错误信息
                                vo.setResult(null);
                                final RuleValidator validator = DiConfig.getValidator(vo.getMode());
                                validator.validate(vo);
                                if (StringUtils.isNotBlank(vo.getResult())) {
                                    ExcelVO excelVO = new ExcelVO();
                                    BeanUtils.copyProperties(excels[i], excelVO);
                                    excelVO.setCode(item.getCode())
                                            .setTitle(item.getTitle())
                                            .setValue(item.getValues().get(i))
                                            .setMessage(vo.getResult());
                                    item.addError(excelVO);
                                }
                            }
                        }
                    });
            // 第三步导入临时表
            String sql = "INSERT INTO {}(" + verifyList.stream()
                    .map(VerifyVO::getField).collect(Collectors.joining(","))
                    + ") VALUES ({})";
            String[] sqlArr = new String[length];
            for (int i = 0; i < length; i++) {
                StringBuilder sb = new StringBuilder();
                for (VerifyVO v : verifyList) {
                    sb.append("'").append(v.getValues().get(i)).append("',");
                }
                sb.deleteCharAt(sb.length() - 1);
                sqlArr[i] = StrFormatter.format(sql, template.getTableName() + "_TEMP", sb);
            }
            try {
                DataSourceUtil.switchDs(template.getDataSourceId());
                // 需要先清空临时表的数据
                jdbcTemplate.execute("DELETE FROM " + template.getTableName() + "_TEMP");
                jdbcTemplate.batchUpdate(sqlArr);
            } catch (Exception e) {
                throw new RuntimeException(e);
            } finally {
                DataSourceUtil.switchDefaultDs();
            }
            // 第四步导出异常excel
            List<ExcelVO> list = new ArrayList<>();
            verifyList.forEach(item -> {
                if (!CollectionUtils.isEmpty(item.getErrors())) {
                    list.addAll(item.getErrors());
                }
            });
            ExcelUtil<ExcelVO> util = new ExcelUtil<>(ExcelVO.class);
            util.exportExcel(response, list, "异常数据分析");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public String resetCoordinate(String coordinate, int index) {
        final String[] coords = coordinate.split(",");
        int columnStart = DiConfig.getIndex(coords[0]);
        return DiConfig.getWord(columnStart + index) + "," + coords[1];
    }
}
