package com.tbyf.his.web.controller.dataImport;

import com.baomidou.mybatisplus.annotation.DbType;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.web.dataImport.DataImportUtils;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.domain.param.AddFieldParam;
import com.tbyf.his.web.dataImport.domain.param.QueryFieldParam;
import com.tbyf.his.web.dataImport.domain.param.QueryMetaFieldParam;
import com.tbyf.his.web.dataImport.domain.param.UpdateFieldParam;
import com.tbyf.his.web.dataImport.domain.vo.CreateFieldVO;
import com.tbyf.his.web.dataImport.domain.vo.FieldDbDiffVO;
import com.tbyf.his.web.dataImport.domain.vo.FieldErrorVo;
import com.tbyf.his.web.dataImport.entity.*;
import com.tbyf.his.web.dataImport.mapper.MetaFieldMapper;
import com.tbyf.his.web.dataImport.service.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/16 15:29
 */
@RestController
@Api(tags = "数据字段接口")
@RequestMapping("/data/field")
@Slf4j
public class DataFieldController {

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private DataTemplateService dataTemplateService;

    @Autowired
    private ExcelDataService excelDataService;

    @Autowired
    private MetaFieldService metaFieldService;

    @Autowired
    private DataRuleService dataRuleService;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private MetaFieldMapper metaFieldMapper;

    @IgnoreWebSecurity
    @GetMapping("")
    @ApiOperation("字段查询")
    public TableDataInfo queryField(@Validated QueryFieldParam param) {
        /*final Page<DataField> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class);
        queryWrapper.eq(StringUtils.isNotBlank(param.getTemplateId()), DataField::getTemplateId, param.getTemplateId())
                .like(StringUtils.isNotBlank(param.getCode()), DataField::getCode, param.getCode())
                .like(StringUtils.isNotBlank(param.getTitle()), DataField::getTitle, param.getTitle())
                .like(StringUtils.isNotBlank(param.getCoordinate()), DataField::getCoordinate, param.getCoordinate())
                .like(StringUtils.isNotBlank(param.getField()), DataField::getField, param.getField());
        final Page<DataField> templatePage = dataFieldService.page(page, queryWrapper);*/
        return param.convert(dataFieldService.queryField(param));
    }

    @IgnoreWebSecurity
    @GetMapping("/dict")
    @ApiOperation("字段查询字典,查询的是当前模板之前年份的同类模板")
    public AjaxResult queryFieldDict(@RequestParam String templateId) {
        final DataTemplate template = dataTemplateService.getById(templateId);
        if (template != null) {
            final DataTemplate dataTemplate = dataTemplateService.getOne(Wrappers.lambdaQuery(DataTemplate.class)
                    .eq(DataTemplate::getOrgName, template.getOrgName())
                    .lt(DataTemplate::getYear, template.getYear())
                    .orderByDesc(DataTemplate::getYear), false);
            if (dataTemplate != null) {
                return AjaxResult.success(dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                        .eq(DataField::getTemplateId, dataTemplate.getId())
                        .isNotNull(DataField::getField)
                        .orderByAsc(DataField::getSort)));
            }
        }
        return AjaxResult.success(Collections.emptyList());
    }

    @IgnoreWebSecurity
    @PostMapping("")
    @ApiOperation("字段新增")
    public AjaxResult addField(@RequestBody @Validated AddFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.save(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/update")
    @ApiOperation("修改字段")
    public AjaxResult updateField(@RequestBody @Validated UpdateFieldParam param) {
        DataField field = new DataField();
        BeanUtils.copyProperties(param, field);
        dataFieldService.updateById(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/delete")
    @ApiOperation("删除字段")
    public AjaxResult deleteField(@RequestParam String fieldId) {
        dataFieldService.removeById(fieldId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, fieldId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/reset")
    @ApiOperation("根据基础模板重置字段")
    public AjaxResult resetField(@RequestParam String excelId, @RequestParam(required = false) String year) {
        final ExcelData excelData = excelDataService.getById(excelId);
        // 获取需要同步的字段列表
        List<DataField> fieldMatchList = null;
        if (StringUtils.isNotBlank(year)) {
            DataTemplate template = dataTemplateService.getOne(Wrappers.lambdaQuery(DataTemplate.class)
                    .eq(DataTemplate::getYear, year).eq(DataTemplate::getOrgName, excelData.getOrgName()), false);
            fieldMatchList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                    .eq(DataField::getTemplateId, template.getId()).isNotNull(DataField::getField)
                    .ne(DataField::getField, ""));
        }
        // 删除掉之前的模板信息
        final LambdaQueryWrapper<DataField> wrapper = Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, excelData.getTemplateId());
        final List<DataField> list = dataFieldService.list(wrapper);
        if (!CollectionUtils.isEmpty(list)) {
            final List<String> fieldIdList = list.stream().map(DataField::getId).collect(Collectors.toList());
            bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).in(BindRule::getDataId, fieldIdList));
            // 模板规则也要删除
            List<BindRule> templateRuleList = bindRuleService.list(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, excelData.getTemplateId()));
            if (!CollectionUtils.isEmpty(templateRuleList)) {
                dataRuleService.remove(Wrappers.lambdaQuery(DataRule.class).in(DataRule::getId, templateRuleList.stream().map(BindRule::getRuleId).collect(Collectors.toList())));
            }
            bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).in(BindRule::getDataId, Collections.singletonList(excelData.getTemplateId())));
        }
        dataFieldService.remove(Wrappers.lambdaQuery(DataField.class).eq(DataField::getTemplateId, excelData.getTemplateId()));
        try (InputStream is = new ByteArrayInputStream(excelData.getFile()); Workbook workbook = WorkbookFactory.create(is)) {
            final Sheet sheet = workbook.getSheetAt(0);
            final int rows = sheet.getLastRowNum() + 1;
            List<DataField> fieldList = new ArrayList<>();
            for (int i = 1; i < rows; i++) {
                final Row row = sheet.getRow(i);
                String code = DiConfig.getValue(row.getCell(1, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL));
                String title = DiConfig.getValue(row.getCell(2, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL));
                String unit = DiConfig.getValue(row.getCell(4, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL));
                if (StringUtils.isAllBlank(code, title, unit)) {
                    continue;
                }
                if (StringUtils.equals(code, "代码") || StringUtils.equals(title, "指标名称") || StringUtils.equals(unit, "计量单位")) {
                    continue;
                }
                final DataField dataField = new DataField();
                dataField.setTemplateId(excelData.getTemplateId());
                dataField.setCode(code);
                dataField.setTitle(title);
                dataField.setUnit(unit);
                // TODO 这里坐标修改为i+1是因为excel文件中展示的第一行号是1,符合直觉
                dataField.setCoordinate("F," + (i + 1));
                dataField.setSort(i + 1);
                // 字段绑定确定进行精确匹配
                dataFieldService.fieldMatch(dataField, fieldMatchList);
                fieldList.add(dataField);
            }
            dataFieldService.saveBatch(fieldList);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/meta")
    @ApiOperation("元字段查询")
    public TableDataInfo queryField(@Validated QueryMetaFieldParam param) {
        final Page<MetaField> page = Page.of(param.getPageNum(), param.getPageSize());
        final QueryWrapper<MetaField> queryWrapper = new QueryWrapper<>();
        //final LambdaQueryWrapper<MetaField> queryWrapper = Wrappers.lambdaQuery(MetaField.class);
        queryWrapper.like(StringUtils.isNotBlank(param.getFieldName()), "field_name", param.getFieldName())
                .like(StringUtils.isNotBlank(param.getFieldType()), "field_type", param.getFieldType())
                .like(StringUtils.isNotBlank(param.getFieldComment()), "field_comment", param.getFieldComment());
        if (StringUtils.equals("length", param.getErrorType())) {
            queryWrapper.gt("length(field_name)", 30);
        }
        final Page<MetaField> templatePage = metaFieldService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @PostMapping("/meta")
    @ApiOperation("元字段新增")
    public AjaxResult addMetaField(@RequestBody @Validated MetaField field) {
        field.setFieldType(field.getFieldType().toUpperCase());
        final LambdaQueryWrapper<MetaField> wrapper = Wrappers.lambdaQuery(MetaField.class)
                .eq(MetaField::getFieldName, field.getFieldName());
        final long count = metaFieldService.count(wrapper);
        if (count > 0) {
            return AjaxResult.error("不能新增相同名称的字段");
        }
        metaFieldService.save(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/meta/update")
    @ApiOperation("修改元字段")
    public AjaxResult updateMeatField(@RequestBody @Validated MetaField field) {
        if (StringUtils.isNotBlank(field.getFieldType())) {
            field.setFieldType(field.getFieldType().toUpperCase());
        }
        metaFieldService.updateById(field);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/meta/delete")
    @ApiOperation("删除元字段")
    public AjaxResult deleteMetaField(@RequestParam String fieldId) {
        metaFieldService.removeById(fieldId);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/meta/dict")
    @ApiOperation("查询元字段字典")
    public AjaxResult getMetaFieldDict() {
        return AjaxResult.success(metaFieldService.list());
    }

    @IgnoreWebSecurity
    @GetMapping("/querySyncField")
    @ApiOperation("字段同步查询")
    public AjaxResult querySyncField(@RequestParam String templateId) {
        final DataTemplate template = dataTemplateService.getById(templateId);
        final List<CreateFieldVO> fields = dataFieldService.getCreateFields(template.getId());
        try {
            final DbType dbType = dataTemplateService.getDbType(template.getDataSourceId());
            DataSourceService.switchDb(template.getDataSourceId());
            String sql = StrFormatter.format("SELECT UTC.COLUMN_NAME AS VALUE,UTC.DATA_TYPE AS TYPE,( SELECT  UCC.COMMENTS  FROM user_col_comments UCC WHERE UCC.COLUMN_NAME = UTC.COLUMN_NAME AND UCC.TABLE_NAME = UTC.TABLE_NAME ) AS LABEL FROM user_tab_columns UTC WHERE UTC.TABLE_NAME = '{}'", template.getTableName());
            if (dbType.equals(DbType.MYSQL)) {
                sql = StrFormatter.format("SELECT column_name AS VALUE, DATA_TYPE AS TYPE, column_comment AS LABEL FROM information_schema.COLUMNS WHERE table_name = '{}'", template.getTableName());
            }
            final List<Map<String, Object>> maps = jdbcTemplate.queryForList(sql);
            final List<String> collect = maps.stream().map(item -> DataImportUtils.str(item.get("VALUE"))).collect(Collectors.toList());
            return AjaxResult.success(fields.stream().filter(item -> !collect.contains(item.getFieldName())).collect(Collectors.toList()));
        } finally {
            DataSourceService.switchDefault();
        }
    }

    @IgnoreWebSecurity
    @PostMapping("/syncField")
    @ApiOperation("字段同步操作")
    public AjaxResult syncField(@RequestBody List<CreateFieldVO> list) {
        final String fieldId = list.get(0).getId();
        final DataField dataField = dataFieldService.getById(fieldId);
        final DataTemplate dataTemplate = dataTemplateService.getById(dataField.getTemplateId());
        List<String> sqlList = new ArrayList<>();
        String comment = "COMMENT ON COLUMN {}.{} IS '{}'";
        String alter = "ALTER TABLE {} ADD {} {} ";
        list.forEach(item -> {
            sqlList.add(StrFormatter.format(alter, dataTemplate.getTableName(), item.getFieldName(), item.getFieldType()));
            sqlList.add(StrFormatter.format(comment, dataTemplate.getTableName(), item.getFieldName(), StringUtils.isBlank(item.getFieldComment()) ? item.getTitle() : item.getFieldComment()));
            sqlList.add(StrFormatter.format(alter, dataTemplate.getTableName() + "_TEMP", item.getFieldName(), item.getFieldType()));
            sqlList.add(StrFormatter.format(comment, dataTemplate.getTableName() + "_TEMP", item.getFieldName(), StringUtils.isBlank(item.getFieldComment()) ? item.getTitle() : item.getFieldComment()));
        });
        String[] sqlArr = new String[sqlList.size()];
        sqlList.toArray(sqlArr);
        try {
            DataSourceService.switchDb(dataTemplate.getDataSourceId());
            jdbcTemplate.batchUpdate(sqlArr);
        } finally {
            DataSourceService.switchDefault();
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/sameFieldCheck")
    @ApiOperation("相同字段检查")
    public AjaxResult sameFieldCheck(@RequestParam String templateId) {
        List<CreateFieldVO> fieldList = dataFieldService.getCreateFields(templateId);
        if (CollectionUtils.isEmpty(fieldList)) {
            return AjaxResult.success(Collections.emptyList());
        }
        Map<String, Integer> map = new HashMap<>();
        List<CreateFieldVO> list = new ArrayList<>();
        for (CreateFieldVO field : fieldList) {
            if (map.containsKey(field.getFieldName())) {
                map.put(field.getFieldName(), map.get(field.getFieldName()) + 1);
            } else {
                map.put(field.getFieldName(), 1);
            }
        }
        map.forEach((key, value) -> {
            if (value > 1) {
                Optional<CreateFieldVO> first = fieldList.stream().filter(item -> StringUtils.equals(item.getFieldName(), key)).findFirst();
                first.ifPresent(list::add);
            }
        });
        return AjaxResult.success(list);
    }


    @IgnoreWebSecurity
    //@GetMapping("/testQuick")
    //@ApiOperation("查询原来模板的字段")
    public AjaxResult testQuick() {
        try {
            DataSourceService.switchDb("datacenter");
            final List<String> list = Arrays.asList("T0801_SERVERCENTER", "T0801_HEALTHCENTER",
                    "T0801_SERSTATION", "T0801_CLINIC", "T0801_TCMHOSP");
            String sql = "SELECT UCO.TABLE_NAME ,UCO.COMMENTS, UCO.COLUMN_NAME,UTO.DATA_TYPE,UTO.DATA_LENGTH,UTO.DATA_PRECISION,UTO.DATA_SCALE FROM\n" +
                    "USER_COL_COMMENTS UCO LEFT JOIN USER_TAB_COLUMNS UTO ON UCO.COLUMN_NAME = UTO.COLUMN_NAME\n" +
                    "WHERE UCO.TABLE_NAME = UPPER('{}') AND UTO.TABLE_NAME = UPPER('{}')";
            List<MetaField> metaFieldList = new ArrayList<>();
            list.forEach(item -> {
                final List<Map<String, Object>> mapList = jdbcTemplate.queryForList(StrFormatter.format(sql, item, item));
                mapList.forEach(v -> {
                    final MetaField metaField = new MetaField();
                    metaField.setRemarks(String.valueOf(v.get("TABLE_NAME")));
                    metaField.setFieldName(String.valueOf(v.get("COLUMN_NAME")));
                    String fieldType = String.valueOf(v.get("DATA_TYPE"));
                    if (Objects.isNull(v.get("DATA_PRECISION"))) {
                        metaField.setFieldType(fieldType + "(" + v.get("DATA_LENGTH") + ")");
                    } else {
                        fieldType = fieldType + "(" +
                                v.get("DATA_PRECISION") +
                                (Objects.isNull(v.get("DATA_SCALE")) ? ")" : "," + v.get("DATA_SCALE") + ")");
                        metaField.setFieldType(fieldType);
                    }
                    metaField.setFieldComment(StringUtils.trim((String) v.get("COMMENTS")));
                    metaFieldList.add(metaField);
                });
            });
            List<MetaField> fieldList = metaFieldList.stream()
                    .filter(DataImportController.distinctByKey(MetaField::getFieldName))
                    .distinct().collect(Collectors.toList());
            DataSourceService.switchDefault();
            metaFieldService.saveBatch(fieldList);
        } finally {
            DataSourceService.switchDefault();
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/fieldDbDiff")
    @ApiOperation("字段与数据库差异化查询")
    public AjaxResult fieldDbDiff(@RequestBody MetaField metaField) {
        // 查询绑定了此字段的所有模板
        LambdaQueryWrapper<DataField> wrapper = Wrappers.lambdaQuery(DataField.class).eq(DataField::getField, metaField.getId());
        List<DataField> fieldList = dataFieldService.list(wrapper);
        if (CollectionUtils.isEmpty(fieldList)) {
            return AjaxResult.success(Collections.emptyList());
        }
        LambdaQueryWrapper<DataTemplate> queryWrapper = Wrappers.lambdaQuery(DataTemplate.class).in(DataTemplate::getId, fieldList.stream().map(DataField::getTemplateId).toArray());
        List<DataTemplate> templateList = dataTemplateService.list(queryWrapper);
        if (CollectionUtils.isEmpty(fieldList)) {
            return AjaxResult.success(Collections.emptyList());
        }
        List<FieldDbDiffVO> list = new ArrayList<>();
        templateList.forEach(item -> {
            Optional<FieldDbDiffVO> first = list.stream().filter(l -> StringUtils.equalsIgnoreCase(item.getDataSourceId(), item.getTableName())).findFirst();
            if (!first.isPresent()) {
                FieldDbDiffVO vo = new FieldDbDiffVO();
                vo.setDatasourceId(item.getDataSourceId());
                vo.setTableName(item.getTableName().toUpperCase());
                vo.setColumnName(metaField.getFieldName().toUpperCase());
                vo.setMetaType(metaField.getFieldType());
                list.add(vo);

                FieldDbDiffVO vo2 = new FieldDbDiffVO();
                vo2.setDatasourceId(item.getDataSourceId());
                vo2.setTableName(item.getTableName().toUpperCase() + "_TEMP");
                vo2.setColumnName(metaField.getFieldName().toUpperCase());
                vo2.setMetaType(metaField.getFieldType());
                list.add(vo2);
            }
        });
        if (CollectionUtils.isEmpty(list)) {
            return AjaxResult.success(Collections.emptyList());
        }
        return AjaxResult.success(list.stream().peek(item -> {
                    try {
                        DataSourceService.switchDb(item.getDatasourceId());
                        List<FieldDbDiffVO> vos = metaFieldMapper.selectFieldDbDiffVO(item.getTableName().toUpperCase(), item.getColumnName().toUpperCase());
                        if (!CollectionUtils.isEmpty(vos)) {
                            FieldDbDiffVO diffVO = vos.get(0);
                            item.setColumnType(diffVO.getColumnType());
                            item.setColumnLength(diffVO.getColumnLength());
                            item.setColumnPrecision(diffVO.getColumnPrecision());
                            item.setColumnScale(diffVO.getColumnScale());
                            item.setMergeSql(StrFormatter.format("ALTER TABLE {} MODIFY ( {} {} )",
                                    item.getTableName().toUpperCase(), item.getColumnName().toUpperCase(), item.getMetaType().toUpperCase()));
                        }
                    } catch (Exception ignored) {
                    } finally {
                        DataSourceService.switchDefault();
                    }
                }).filter(item -> StringUtils.isNotBlank(item.getColumnType()))
                .collect(Collectors.toList()));
    }

    @IgnoreWebSecurity
    @PostMapping("/executeSql")
    @ApiOperation("sql执行")
    public AjaxResult executeSql(@RequestBody FieldDbDiffVO vo) {
        try {
            DataSourceService.switchDb(vo.getDatasourceId());
            jdbcTemplate.execute(vo.getMergeSql());
        } finally {
            DataSourceService.switchDefault();
        }
        return AjaxResult.success("同步成功");
    }

}
