package com.tbyf.his.web.controller.dataImport;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tbyf.his.common.annotation.IgnoreWebSecurity;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.bean.BeanUtils;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.core.RuleVO;
import com.tbyf.his.web.dataImport.core.RuleValidator;
import com.tbyf.his.web.dataImport.domain.param.AddRuleParam;
import com.tbyf.his.web.dataImport.domain.param.BindRuleParam;
import com.tbyf.his.web.dataImport.domain.param.QueryRuleParam;
import com.tbyf.his.web.dataImport.domain.param.UpdateRuleParam;
import com.tbyf.his.web.dataImport.entity.BindRule;
import com.tbyf.his.web.dataImport.entity.DataField;
import com.tbyf.his.web.dataImport.entity.DataRule;
import com.tbyf.his.web.dataImport.entity.DataTemplate;
import com.tbyf.his.web.dataImport.service.BindRuleService;
import com.tbyf.his.web.dataImport.service.DataFieldService;
import com.tbyf.his.web.dataImport.service.DataRuleService;
import com.tbyf.his.web.dataImport.service.DataTemplateService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/16 15:29
 */
@RestController
@Api(tags = "数据字段接口")
@RequestMapping("/data/rule")
@Slf4j
public class DataRuleController {

    @Autowired
    private DataRuleService dataRuleService;

    @Autowired
    private BindRuleService bindRuleService;

    @Autowired
    private DataFieldService dataFieldService;

    @Autowired
    private DataTemplateService dataTemplateService;

    @IgnoreWebSecurity
    @GetMapping("")
    @ApiOperation("规则查询")
    public TableDataInfo queryRule(@Validated QueryRuleParam param) {
        final Page<DataRule> page = Page.of(param.getPageNum(), param.getPageSize());
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.like(StringUtils.isNotBlank(param.getName()), DataRule::getName, param.getName())
                .eq(StringUtils.isNotBlank(param.getType()), DataRule::getType, param.getType())
                .in(DataRule::getType, "基础规则", "组合规则");
        final Page<DataRule> templatePage = dataRuleService.page(page, queryWrapper);
        return param.convert(templatePage);
    }

    @IgnoreWebSecurity
    @GetMapping("/dict")
    @ApiOperation("规则字典")
    public AjaxResult queryRuleDict() {
        final LambdaQueryWrapper<DataRule> queryWrapper = Wrappers.lambdaQuery(DataRule.class);
        queryWrapper.in(DataRule::getType, "基础规则", "组合规则")
                .select(DataRule::getId, DataRule::getName);
        return AjaxResult.success(dataRuleService.list(queryWrapper));
    }

    @IgnoreWebSecurity
    @PostMapping("")
    @ApiOperation("规则新增")
    public AjaxResult addRule(@RequestBody @Validated AddRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.save(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/update")
    @ApiOperation("修改规则")
    public AjaxResult updateRule(@RequestBody @Validated UpdateRuleParam param) {
        DataRule rule = new DataRule();
        BeanUtils.copyProperties(param, rule);
        dataRuleService.updateById(rule);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/delete")
    @ApiOperation("删除规则")
    public AjaxResult deleteRule(@RequestParam String ruleId) {
        dataRuleService.removeById(ruleId);
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getRuleId, ruleId));
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @PostMapping("/test")
    @ApiOperation("规则测试")
    public AjaxResult testRule(@RequestBody @Validated RuleVO test) {
        final RuleValidator validator = DiConfig.getValidator(test.getMode());
        validator.validate(test);
        if (StringUtils.isNotBlank(test.getResult())) {
            return AjaxResult.error(test.getResult());
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind/quick")
    @ApiOperation("快速绑定")
    public AjaxResult quickBind(@RequestParam String templateId,
                                @RequestParam String ruleId) {
        final LambdaQueryWrapper<DataField> queryWrapper = Wrappers.lambdaQuery(DataField.class)
                .select(DataField::getId)
                .eq(DataField::getTemplateId, templateId);
        final List<String> fieldIdList = dataFieldService.list(queryWrapper).stream()
                .map(DataField::getId)
                .collect(Collectors.toList());
        final LambdaQueryWrapper<BindRule> removeWrapper = Wrappers.lambdaQuery(BindRule.class)
                .in(BindRule::getDataId, fieldIdList)
                .eq(BindRule::getRuleId, ruleId);
        bindRuleService.remove(removeWrapper);
        final List<BindRule> bindRuleList = fieldIdList.stream().map(item -> {
            BindRule bind = new BindRule();
            bind.setDataId(item);
            bind.setRuleId(ruleId);
            return bind;
        }).collect(Collectors.toList());
        bindRuleService.saveBatch(bindRuleList);
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/bind")
    @ApiOperation("查询绑定的规则")
    public AjaxResult queryBindRule(@RequestParam String fieldId) {
        final List<BindRule> list = bindRuleService.list(Wrappers.lambdaQuery(BindRule.class)
                .eq(BindRule::getDataId, fieldId));
        if (CollectionUtils.isEmpty(list)) {
            return AjaxResult.success(Collections.emptyList());
        }
        final List<String> idList = list.stream().map(BindRule::getRuleId).collect(Collectors.toList());
        return AjaxResult.success(dataRuleService.list(Wrappers.lambdaQuery(DataRule.class)
                .in(DataRule::getId, idList)));
    }

    @IgnoreWebSecurity
    @PostMapping("/bind")
    @ApiOperation("配置字段规则绑定")
    @Transactional(rollbackFor = Exception.class)
    public AjaxResult bindRule(@RequestBody BindRuleParam param) {
        bindRuleService.remove(Wrappers.lambdaQuery(BindRule.class).eq(BindRule::getDataId, param.getFieldId()));
        if (!CollectionUtils.isEmpty(param.getRuleIdList())) {
            final List<BindRule> bindRuleList = param.getRuleIdList().stream().map(item -> {
                BindRule bd = new BindRule();
                bd.setRuleId(item);
                bd.setDataId(param.getFieldId());
                return bd;
            }).collect(Collectors.toList());
            bindRuleService.saveBatch(bindRuleList);
        }
        return AjaxResult.success();
    }

    @IgnoreWebSecurity
    @GetMapping("/sync/year")
    @ApiOperation("根据年份同步规则")
    public AjaxResult syncRuleByYear(@RequestParam String templateId, @RequestParam String year) {
        // 查询两个模板 source-本模板 target-对比同步的模板
        DataTemplate sourceTemplate = dataTemplateService.getById(templateId);
        DataTemplate targetTemplate = dataTemplateService.getOne(Wrappers.lambdaQuery(DataTemplate.class).eq(DataTemplate::getYear, year).eq(DataTemplate::getOrgName, sourceTemplate.getOrgName()), false);
        // 查询本模板的字段
        List<DataField> fieldList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .eq(DataField::getTemplateId, sourceTemplate.getId())
                .isNotNull(DataField::getField)
                .select(DataField::getField, DataField::getId));
        // 查询被同步的模板的字段
        List<DataField> targetFieldList = dataFieldService.list(Wrappers.lambdaQuery(DataField.class)
                .eq(DataField::getTemplateId, targetTemplate.getId())
                .isNotNull(DataField::getField)
                .select(DataField::getField, DataField::getId));
        Set<String> fieldNameSet = fieldList.stream().map(DataField::getField).collect(Collectors.toSet());
        if (!CollectionUtils.isEmpty(fieldList)) {
            List<BindRule> rules = new ArrayList<>();
            // 查询两个模板各自的字段规则
            List<BindRule> sourceBindList = bindRuleService.queryRuleList(sourceTemplate.getId());
            List<BindRule> targetBindList = bindRuleService.queryRuleList(targetTemplate.getId());
            // 字段规则同步
            if (!CollectionUtils.isEmpty(targetBindList)) {
                // 遍历字段规则
                for (BindRule targetRule : targetBindList) {
                    // 匹配此规则对应的字段ID是否在本模板中存在
                    Optional<DataField> first = fieldList.stream().filter(df -> StringUtils.equals(df.getField(), targetRule.getDataId())).findFirst();
                    if (first.isPresent()) {
                        // 判断本模板是否存在相同的字段及规则
                        boolean match = sourceBindList.stream()
                                .anyMatch(br -> StringUtils.equals(br.getDataId(), targetRule.getDataId()) && StringUtils.equals(br.getRuleId(), targetRule.getRuleId()));
                        if (!match) {
                            rules.add(new BindRule(first.get().getId(), targetRule.getRuleId()));
                        }
                    }
                }
            }
            // 模板规则同步
            List<DataRule> sourceRules = dataRuleService.queryTemplateRules(sourceTemplate.getId());
            List<DataRule> targetRules = dataRuleService.queryTemplateRules(targetTemplate.getId());
            if (!CollectionUtils.isEmpty(targetRules)) {
                List<DataRule> datas = new ArrayList<>();
                root:
                for (DataRule targetRule : targetRules) {
                    String[] split = targetRule.getContent().split("=");
                    // 查询出对应的两边的字段ID
                    List<String> list1 = Arrays.asList(split[0].split(","));
                    List<String> list2 = Arrays.asList(split[1].split(","));
                    // 先判断其中的每个字段是否在源表中存在
                    boolean match1 = fieldNameSet.containsAll(list1);
                    boolean match2 = fieldNameSet.containsAll(list2);
                    if (!match1 || !match2){
                        continue root;
                    }
                    // 再判断此规则是否已经在本模板中存在
                    boolean match = sourceRules.stream().anyMatch(item -> {
                        String[] temp = item.getContent().split("=");
                        // 查询出对应的两边的字段ID
                        List<String> temp1 = Arrays.asList(temp[0].split(","));
                        List<String> temp2 = Arrays.asList(temp[1].split(","));
                        if (temp1.containsAll(list1) && temp2.containsAll(list2)){
                            return true;
                        }
                        return temp1.containsAll(list2) && temp2.containsAll(list1);
                    });
                    if (!match){
                        targetRule.setId(null);
                        datas.add(targetRule);
                    }
                }
                if (!CollectionUtils.isEmpty(datas)) {
                    dataRuleService.saveBatch(datas);
                    datas.forEach(item -> {
                        BindRule rule = new BindRule();
                        rule.setDataId(sourceTemplate.getId())
                                .setRuleId(item.getId());
                        rules.add(rule);
                    });
                }
            }
            if (!CollectionUtils.isEmpty(rules)) {
                bindRuleService.saveBatch(rules);
            }
        }
        return AjaxResult.success();
    }


}
