package com.tbyf.his.web.controller.tool;

import com.tbyf.his.common.annotation.Log;
import com.tbyf.his.common.core.controller.BaseController;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.enums.BusinessType;
import com.tbyf.his.common.utils.poi.ExcelUtil;
import com.tbyf.his.tool.domain.DynamicForm;
import com.tbyf.his.tool.service.IDynamicFormService;
import com.tbyf.his.tool.vo.DynamicTableSave;
import com.tbyf.his.tool.vo.ImportData;
import com.tbyf.his.tool.vo.TableDataSearch;
import com.tbyf.his.tool.vo.TableDataVO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * @author: fr
 * @date: 2022年08月17日 11:38
 */
@RestController
@RequestMapping("/tool/dynamicForm")
public class DynamicFormController extends BaseController {
    @Autowired
    private IDynamicFormService dynamicFormService;

    /**
     * 查询动态单列表
     */
    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:list')")
    @GetMapping("/list")
    public TableDataInfo list(DynamicForm dynamicForm) {
        startPage();
        List<DynamicForm> list = dynamicFormService.selectDynamicFormList(dynamicForm);
        return getDataTable(list);
    }

    /**
     * 导出动态单列表
     */
    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:export')")
    @Log(title = "动态单", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, DynamicForm dynamicForm) {
        List<DynamicForm> list = dynamicFormService.selectDynamicFormList(dynamicForm);
        ExcelUtil<DynamicForm> util = new ExcelUtil<DynamicForm>(DynamicForm.class);
        util.exportExcel(response, list, "动态单数据");
    }

    /**
     * 获取动态单详细信息
     */
    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:getInfo')")
    @GetMapping(value = "/getInfo/{formId}")
    public AjaxResult getInfo(@PathVariable("formId") Long formId) {
        return AjaxResult.success(dynamicFormService.selectDynamicFormByFormId(formId));
    }

    /**
     * 新增动态单
     */
    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:add')")
    @Log(title = "动态单", businessType = BusinessType.INSERT)
    @PostMapping("/addForm")
    public AjaxResult add(@RequestBody DynamicForm dynamicForm) {
        dynamicForm.setCreateBy(getUsername());
        dynamicForm.setDeptId(getDeptId());
        return toAjax(dynamicFormService.insertDynamicForm(dynamicForm));
    }

    /**
     * 修改动态单
     */
    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:edit')")
    @Log(title = "动态单", businessType = BusinessType.UPDATE)
    @PutMapping("/updateForm")
    public AjaxResult edit(@RequestBody DynamicForm dynamicForm) {
        dynamicForm.setUpdateBy(getUsername());
        return toAjax(dynamicFormService.updateDynamicForm(dynamicForm));
    }

    /**
     * 删除动态单
     */
    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:remove')")
    @Log(title = "动态单", businessType = BusinessType.DELETE)
    @DeleteMapping(value = {"/{formId}/{datasourceId}/{tableName}", "/{formId}/{datasourceId}"})
    public AjaxResult remove(@PathVariable("formId") Long formId,
                             @PathVariable("datasourceId") Long datasourceId,
                             @PathVariable(value = "tableName", required = false) String tableName) {
        dynamicFormService.deleteDynamicFormByFormId(formId, datasourceId, tableName);
        return toAjax(1);
    }

    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:dynamicList')")
    @PostMapping("/dynamicList")
    public TableDataVO dynamicList(@RequestBody TableDataSearch search) {
        return dynamicFormService.dynamicList(search);
    }


    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:dynamicAdd')")
    @Log(title = "动态单表新增", businessType = BusinessType.INSERT)
    @PostMapping("/dynamicAdd")
    public AjaxResult dynamicAdd(@RequestBody DynamicTableSave data) {
        dynamicFormService.dynamicAdd(data);
        return toAjax(1);
    }

    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:dynamicEdit')")
    @Log(title = "动态单表编辑", businessType = BusinessType.UPDATE)
    @PostMapping("/dynamicEdit")
    public AjaxResult dynamicEdit(@RequestBody DynamicTableSave data) {
        dynamicFormService.dynamicEdit(data);
        return toAjax(1);
    }

    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:dynamicImport')")
    @Log(title = "动态表单导入", businessType = BusinessType.IMPORT)
    @PostMapping("/dynamicImport")
    public AjaxResult dynamicImport(@RequestBody ImportData data) {
        dynamicFormService.dynamicImport(data.getList(), data.getIsCover());
        return toAjax(1);
    }

    @PreAuthorize("@ss.hasPermi('tool:dynamicForm:dynamicDelete')")
    @Log(title = "动态表单删除", businessType = BusinessType.DELETE)
    @PostMapping("/dynamicDelete")
    public AjaxResult dynamicDelete(@RequestBody DynamicTableSave data) {
        dynamicFormService.dynamicDelete(data);
        return toAjax(1);
    }


}
