package com.tbyf.his.web.dataImport.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.poi.ExcelUtil;
import com.tbyf.his.web.dataImport.DataImportUtils;
import com.tbyf.his.web.dataImport.core.DiConfig;
import com.tbyf.his.web.dataImport.core.DiConstants;
import com.tbyf.his.web.dataImport.core.RuleVO;
import com.tbyf.his.web.dataImport.core.RuleValidator;
import com.tbyf.his.web.dataImport.domain.vo.ExcelVO;
import com.tbyf.his.web.dataImport.domain.vo.VerifyVO;
import com.tbyf.his.web.dataImport.entity.DataRule;
import com.tbyf.his.web.dataImport.entity.DataTemplate;
import com.tbyf.his.web.dataImport.entity.ExcelData;
import com.tbyf.his.web.dataImport.mapper.ExcelDataMapper;
import com.tbyf.his.web.dataImport.service.DataSourceService;
import com.tbyf.his.web.dataImport.service.DataTemplateService;
import com.tbyf.his.web.dataImport.service.ExcelDataService;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author lzz
 * @date 2023/2/7 11:24
 */
@Slf4j
@Service
@DataSource(DataSourceType.MASTER)
public class ExcelDataServiceImpl extends ServiceImpl<ExcelDataMapper, ExcelData> implements ExcelDataService {

    @Autowired
    private DataTemplateService dataTemplateService;

    @Autowired
    private JdbcTemplate jdbcTemplate;


    @Override
    public void analyzeExport(ExcelData excelData, HttpServletResponse response) {
        final DataTemplate template = dataTemplateService.getById(excelData.getTemplateId());
        List<VerifyVO> verifyList = dataTemplateService.getVerify(template.getId());
        try (InputStream is = new ByteArrayInputStream(excelData.getFile()); Workbook workbook = WorkbookFactory.create(is)) {
            final Sheet sheet = workbook.getSheetAt(0);
            //final int rows = sheet.getLastRowNum() + 1;
            final Optional<VerifyVO> first = verifyList.stream()
                    .filter(item -> StringUtils.isNotBlank(item.getCoordinate()))
                    .findFirst();
            // 数据长度/条数
            final List<Integer> columnList = DiConfig.getLength(first.get(), sheet);
            final int length = columnList.size();
            // 第一步解析数据到values
            for (int a = 0; a < verifyList.size(); a++) {
                final VerifyVO vo = verifyList.get(a);
                if (StringUtils.isBlank(vo.getCoordinate())) {
                    final String s = DiConfig.getCoordinate(vo, sheet);
                    if (StringUtils.isBlank(s)) {
                        vo.addError(new ExcelVO("没有符合代码与指标名称的字段行", ""));
                        continue;
                    }
                    vo.setCoordinate(s);
                }
                final String[] coords = vo.getCoordinate().split(",");
                // TODO 这里减1的原因是excel表名的行号1实际上的下标为0
                int rowStart = Integer.parseInt(coords[1]) - 1;
                final Row row = sheet.getRow(rowStart);
                for (Integer column : columnList) {
                    try {
                        final Cell cell = row.getCell(column, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
                        vo.addValue(DiConfig.getValue(cell));
                    } catch (Exception ignore) {
                    }
                }
            }
            // 获取每一步的核心机构等信息
            ExcelVO[] excels = new ExcelVO[length];
            for (int e = 0; e < length; e++) {
                excels[e] = new ExcelVO();
                excels[e].setOrgType(template.getOrgName());
                for (VerifyVO v : verifyList) {
                    if (StringUtils.equals("ORG_NAME", v.getFieldName())) {
                        excels[e].setOrgName(v.getValues().get(e));
                        continue;
                    }
                    if (StringUtils.equals("ORG_CODE", v.getFieldName())) {
                        excels[e].setOrgCode(v.getValues().get(e));
                        continue;
                    }
                    if (StringUtils.equals("UNIFORM_CREDIT_NUMBER", v.getFieldName())) {
                        excels[e].setUnifiedCode(v.getValues().get(e));
                        continue;
                    }
                }
            }
            // 第二步进行数据校验
            verifyList.stream()
                    .filter(item -> !CollectionUtils.isEmpty(item.getValues()) && !CollectionUtils.isEmpty(item.getRules()))
                    .forEach(item -> {
                        for (int i = 0; i < item.getValues().size(); i++) {
                            for (int j = 0; j < item.getRules().size(); j++) {
                                final RuleVO vo = item.getRules().get(j);
                                vo.setValue(item.getValues().get(i));
                                // 置空错误信息
                                vo.setResult(null);
                                // TODO 特殊数据无需做错误校验
                                if (DiConstants.WHITE.contains(vo.getValue())) {
                                    break;
                                }
                                final RuleValidator validator = DiConfig.getValidator(vo.getMode());
                                validator.validate(vo);
                                if (StringUtils.isNotBlank(vo.getResult())) {
                                    ExcelVO excelVO = new ExcelVO();
                                    BeanUtils.copyProperties(excels[i], excelVO);
                                    excelVO.setCode(item.getCode())
                                            .setTitle(item.getTitle())
                                            .setValue(item.getValues().get(i))
                                            .setMessage(vo.getResult());
                                    item.addError(excelVO);
                                }
                            }
                        }
                    });
            List<ExcelVO> list = new ArrayList<>();
            verifyList.forEach(item -> {
                if (!CollectionUtils.isEmpty(item.getErrors())) {
                    list.addAll(item.getErrors());
                }
            });
            // 模板规则的数据校验
            final List<DataRule> allRule = dataTemplateService.getAllRule(template.getId());
            for (DataRule rule : allRule) {
                if (StringUtils.isNotBlank(rule.getContent())) {
                    if (StringUtils.equals(rule.getMode(), "逻辑相加")) {
                        final String[] split = rule.getContent().split("=");
                        if (split.length == 2) {
                            String[] leftIds = split[0].split(",");
                            String[] rightIds = split[1].split(",");
                            final List<String> leftList = Arrays.stream(leftIds).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                            final List<String> rightList = Arrays.stream(rightIds).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                            if (leftList.size() > 0 && rightList.size() > 0) {
                                for (int r = 0; r < length; r++) {
                                    final List<VerifyVO> leftObjs = verifyList.stream().filter(v -> leftList.contains(v.getFieldId())).collect(Collectors.toList());
                                    final List<VerifyVO> rightObjs = verifyList.stream().filter(v -> rightList.contains(v.getFieldId())).collect(Collectors.toList());
                                    try {
                                        BigDecimal left = new BigDecimal("0");
                                        BigDecimal right = new BigDecimal("0");
                                        for (VerifyVO leftVo : leftObjs) {
                                            try {
                                                left = left.add(new BigDecimal(leftVo.getValues().get(r)));
                                            } catch (Exception ignore) {
                                            }
                                        }
                                        for (VerifyVO rightVo : rightObjs) {
                                            try {
                                                right = right.add(new BigDecimal(rightVo.getValues().get(r)));
                                            } catch (Exception ignore) {
                                            }
                                        }
                                        // 这里需要进行四舍五入的精确位数处理
                                        if(StringUtils.isNotBlank(rule.getRemarks())) {
                                            int scale = Integer.parseInt(rule.getRemarks());
                                            left = left.setScale(scale, RoundingMode.UP);
                                            right = right.setScale(scale, RoundingMode.UP);
                                        }
                                        if (left.compareTo(right) != 0) {
                                            ExcelVO excelVO = new ExcelVO();
                                            BeanUtils.copyProperties(excels[r], excelVO);
                                            excelVO.setMessage("逻辑验证错误,正确公式为:" + getFormula(leftObjs, rightObjs));
                                            leftObjs.addAll(rightObjs);
                                            int finalR = r;
                                            final List<ExcelVO> errList = leftObjs.stream().map(item -> {
                                                final ExcelVO temp = new ExcelVO();
                                                temp.setCode(item.getCode())
                                                        .setTitle(item.getTitle())
                                                        .setValue(item.getValues().get(finalR));
                                                return temp;
                                            }).collect(Collectors.toList());
                                            final ExcelVO firstVo = errList.get(0);
                                            firstVo.setMessage(excelVO.getMessage()).setOrgName(excelVO.getOrgName())
                                                    .setOrgType(excelVO.getOrgType()).setOrgCode(excelVO.getOrgCode())
                                                    .setUnifiedCode(excelVO.getUnifiedCode());
                                            list.addAll(errList);
                                        }
                                    } catch (Exception e) {
                                        ExcelVO excelVO = new ExcelVO();
                                        BeanUtils.copyProperties(excels[r], excelVO);
                                        excelVO.setMessage("逻辑验证错误,正确公式为:" + getFormula(leftObjs, rightObjs));
                                        leftObjs.addAll(rightObjs);
                                        int finalR = r;
                                        final List<ExcelVO> errList = leftObjs.stream().map(item -> {
                                            final ExcelVO temp = new ExcelVO();
                                            temp.setCode(item.getCode())
                                                    .setTitle(item.getTitle())
                                                    .setValue(item.getValues().get(finalR));
                                            return temp;
                                        }).collect(Collectors.toList());
                                        final ExcelVO firstVo = errList.get(0);
                                        firstVo.setMessage(excelVO.getMessage()).setOrgName(excelVO.getOrgName())
                                                .setOrgType(excelVO.getOrgType()).setOrgCode(excelVO.getOrgCode())
                                                .setUnifiedCode(excelVO.getUnifiedCode());
                                        list.addAll(errList);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            // 第三步导入临时表
            String insertSql = "INSERT INTO {}(" + verifyList.stream()
                    .map(VerifyVO::getFieldName).collect(Collectors.joining(","))
                    + ",DORDER,YEAROOFDATARECORD,ROWCODE ) VALUES ({})";
            String[] sqlArr = new String[length];
            for (int i = 0; i < length; i++) {
                StringBuilder sb = new StringBuilder();
                for (VerifyVO v : verifyList) {
                    sb.append(DataImportUtils.getInsertContent(v, i)).append(",");
                }
                sb.append("'").append(DataImportUtils.getNextId()).append("','")
                        .append(template.getYear()).append("年").append("','")
                        .append(DataImportUtils.getNextId()).append("'");
                sqlArr[i] = StrFormatter.format(insertSql, template.getTableName() + "_TEMP", sb);
            }
            try {
                DataSourceService.switchDb(template.getDataSourceId());
                // 需要先清空临时表的数据,按照年份
                jdbcTemplate.execute(StrFormatter.format("DELETE FROM {} WHERE YEAROOFDATARECORD = '{}'"
                        , template.getTableName() + "_TEMP", template.getYear() + "年"));
                jdbcTemplate.batchUpdate(sqlArr);
            } catch (Exception e) {
                throw new RuntimeException(e);
            } finally {
                DataSourceService.switchDefault();
            }
            // 第四步导出异常excel
            ExcelUtil<ExcelVO> util = new ExcelUtil<>(ExcelVO.class);
            util.exportExcel(response, list, "异常数据分析");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取逻辑校验公式
     *
     * @param leftObjs
     * @param rightObjs
     * @return
     */
    private String getFormula(List<VerifyVO> leftObjs, List<VerifyVO> rightObjs) {
        return leftObjs.stream().map(item -> getValue(item.getCode()) + "（" + getValue(item.getTitle()) + "）")
                .collect(Collectors.joining("+"))
                + " = " +
                rightObjs.stream().map(item -> getValue(item.getCode()) + "（" + getValue(item.getTitle()) + "）")
                        .collect(Collectors.joining("+"));
    }

    public String getValue(String s) {
        if (s == null) {
            return "";
        }
        return s;
    }

    public String resetCoordinate(String coordinate, int index) {
        final String[] coords = coordinate.split(",");
        int columnStart = DiConfig.getIndex(coords[0]);
        return DiConfig.getWord(columnStart + index) + "," + coords[1];
    }

    public static void main(String[] args) {
        BigDecimal bigDecimal = new BigDecimal("1234567.123");
        System.out.println(bigDecimal.setScale(5,BigDecimal.ROUND_DOWN).intValue());
    }
}
