package com.tbyf.his.system.service.impl;

import com.tbyf.his.common.constant.UserConstants;
import com.tbyf.his.common.core.domain.TreeSelect;
import com.tbyf.his.common.core.domain.entity.SysUser;
import com.tbyf.his.common.exception.ServiceException;
import com.tbyf.his.common.utils.SecurityUtils;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.spring.SpringUtils;
import com.tbyf.his.system.domain.CategoryTreeSelect;
import com.tbyf.his.system.domain.SysCategory;
import com.tbyf.his.system.mapper.SysCategoryMapper;
import com.tbyf.his.system.service.ISysCategoryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class SysCategoryServiceImpl implements ISysCategoryService {

    @Autowired
    private SysCategoryMapper categoryMapper;

    /**
     * 查询插件种类管理数据
     *
     * @param category 插件种类信息
     * @return 插件种类信息集合
     */
    @Override
    public List<SysCategory> selectDeptList(SysCategory category) {
        return categoryMapper.selectCategoryList(category);
    }

    /**
     * 构建前端所需要树结构
     *
     * @param categories 插件种类列表
     * @return 树结构列表
     */
    @Override
    public List<SysCategory> buildCategoryTree(List<SysCategory> categories) {
        List<SysCategory> returnList = new ArrayList<SysCategory>();
        List<Long> tempList = new ArrayList<Long>();
        for (SysCategory category : categories) {
            tempList.add(category.getCategoryId());
        }
        for (SysCategory category : categories) {
            // 如果是顶级节点, 遍历该父节点的所有子节点
            if (!tempList.contains(category.getParentId())) {
                recursionFn(categories, category);
                returnList.add(category);
            }
        }
        if (returnList.isEmpty()) {
            returnList = categories;
        }
        return returnList;
    }

    @Override
    public List<TreeSelect> buildCategoryTreeSelect(List<SysCategory> categories) {
        List<SysCategory> categoryTrees = buildCategoryTree(categories);
        return categoryTrees.stream().map(CategoryTreeSelect::new).collect(Collectors.toList());
    }

    /**
     * 根据插件种类ID查询信息
     *
     * @param categoryId 插件种类ID
     * @return 插件种类信息
     */
    @Override
    public SysCategory selectCategoryById(Long categoryId) {
        return categoryMapper.selectCategoryById(categoryId);
    }

    /**
     * 根据ID查询所有子种类（正常状态）
     *
     * @param categoryId 部门ID
     * @return 子种类数
     */
    @Override
    public int selectNormalChildrenCategoryById(Long categoryId) {
        return categoryMapper.selectNormalChildrenCategoryById(categoryId);
    }

    /**
     * 是否存在子节点
     *
     * @param categoryId 部门ID
     * @return 结果
     */
    @Override
    public boolean hasChildByCategoryId(Long categoryId) {
        return categoryMapper.hasChildByCategoryId(categoryId) > 0;
    }

    /**
     * 查询插件种类是否存在插件
     *
     * @param categoryId 插件种类ID
     * @return 结果 true 存在 false 不存在
     */
    @Override
    public boolean checkCategoryExistPlugin(Long categoryId) {
        return categoryMapper.checkCategoryExistPlugin(categoryId) > 0;
    }

    /**
     * 校验插件种类名称是否唯一
     *
     * @param category 插件种类信息
     * @return 结果
     */
    @Override
    public String checkCategoryNameUnique(SysCategory category) {
        Long categoryId = StringUtils.isNull(category.getCategoryId()) ? -1L : category.getCategoryId();
        SysCategory info = categoryMapper.checkCategoryNameUnique(category.getCategoryName(), category.getParentId());
        if (StringUtils.isNotNull(info) && info.getCategoryId().longValue() != categoryId.longValue()) {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }

    /**
     * 校验插件种类是否有数据权限
     *
     * @param categoryId 插件种类id
     */
    @Override
    public void checkCategoryDataScope(Long categoryId) {
        if (!SysUser.isAdmin(SecurityUtils.getUserId())) {
            SysCategory category = new SysCategory();
            category.setCategoryId(categoryId);
            List<SysCategory> categories = SpringUtils.getAopProxy(this).selectDeptList(category);
            if (StringUtils.isEmpty(categories)) {
                throw new ServiceException("没有权限访问插件种类数据！");
            }
        }
    }

    /**
     * 新增保存插件种类信息
     *
     * @param category 插件种类信息
     * @return 结果
     */
    @Override
    public int insertDept(SysCategory category) {
        SysCategory info = categoryMapper.selectCategoryById(category.getParentId());
        // 如果父节点不为正常状态,则不允许新增子节点
        /*if (!UserConstants.DEPT_NORMAL.equals(info.getStatus()))
        {
            throw new ServiceException("部门停用，不允许新增");
        }*/
        category.setAncestors(info.getAncestors() + "," + category.getParentId());
        return categoryMapper.insertCategory(category);
    }

    /**
     * 修改保存插件种类信息
     *
     * @param category 插件种类信息
     * @return 结果
     */
    @Override
    public int updateCategory(SysCategory category) {
//        SysCategory newParentCategory = categoryMapper.selectCategoryById(category.getParentId());
//        SysCategory oldCategory = categoryMapper.selectCategoryById(category.getCategoryId());
//        if (StringUtils.isNotNull(newParentCategory) && StringUtils.isNotNull(oldCategory))
//        {
//            String newAncestors = newParentCategory.getAncestors() + "," + newParentCategory.getDeptId();
//            String oldAncestors = oldCategory.getAncestors();
//            category.setAncestors(newAncestors);
//            updateDeptChildren(dept.getDeptId(), newAncestors, oldAncestors);
//        }
//        int result = deptMapper.updateDept(dept);
//        if (UserConstants.DEPT_NORMAL.equals(dept.getStatus()) && StringUtils.isNotEmpty(dept.getAncestors())
//                && !StringUtils.equals("0", dept.getAncestors()))
//        {
//            // 如果该部门是启用状态，则启用该部门的所有上级部门
//            updateParentDeptStatusNormal(dept);
//        }
//        return result;

        return 1;
    }

    @Override
    public int deleteDeptById(Long categoryId) {
        return 0;
    }

    /**
     * 递归列表
     */
    private void recursionFn(List<SysCategory> list, SysCategory t) {
        // 得到子节点列表
        List<SysCategory> childList = getChildList(list, t);
        t.setChildren(childList);
        for (SysCategory tChild : childList) {
            if (hasChild(list, tChild)) {
                recursionFn(list, tChild);
            }
        }
    }

    /**
     * 得到子节点列表
     */
    private List<SysCategory> getChildList(List<SysCategory> list, SysCategory t) {
        List<SysCategory> tlist = new ArrayList<SysCategory>();
        Iterator<SysCategory> it = list.iterator();
        while (it.hasNext()) {
            SysCategory n = (SysCategory) it.next();
            if (StringUtils.isNotNull(n.getParentId()) && n.getParentId().longValue() == t.getCategoryId().longValue()) {
                tlist.add(n);
            }
        }
        return tlist;
    }

    /**
     * 判断是否有子节点
     */
    private boolean hasChild(List<SysCategory> list, SysCategory t) {
        return getChildList(list, t).size() > 0;
    }
}
