package com.tbyf.his.web.controller.system;

import com.tbyf.his.common.annotation.Log;
import com.tbyf.his.common.constant.UserConstants;
import com.tbyf.his.common.core.controller.BaseController;
import com.tbyf.his.common.core.domain.AjaxResult;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.enums.BusinessType;
import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.common.utils.poi.ExcelUtil;
import com.tbyf.his.system.domain.SysPlugin;
import com.tbyf.his.system.service.ISysPluginService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 插件信息 控制层
 */
@RestController
@RequestMapping("/system/plugin")
public class SysPluginController extends BaseController {

    @Autowired
    private ISysPluginService pluginService;

    /**
     * 获取插件列表
     */
    @PreAuthorize("@ss.hasPermi('system:plugin:list')")
    @GetMapping("/list")
    public TableDataInfo list(SysPlugin plugin) {
        startPage();
        List<SysPlugin> list = pluginService.selectPluginList(plugin);
        return getDataTable(list);
    }

    @Log(title = "插件管理", businessType = BusinessType.EXPORT)
    @PreAuthorize("@ss.hasPermi('system:plugin:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysPlugin plugin) {
        List<SysPlugin> list = pluginService.selectPluginList(plugin);
        ExcelUtil<SysPlugin> util = new ExcelUtil<SysPlugin>(SysPlugin.class);
        util.exportExcel(response, list, "插件数据");
    }

    @Log(title = "插件管理", businessType = BusinessType.IMPORT)
    @PreAuthorize("@ss.hasPermi('system:plugin:import')")
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception {
        ExcelUtil<SysPlugin> util = new ExcelUtil<SysPlugin>(SysPlugin.class);
        List<SysPlugin> pluginList = util.importExcel(file.getInputStream());
        String operName = getUsername();
        String message = pluginService.importPlugin(pluginList, updateSupport, operName);
        return AjaxResult.success(message);
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response) {
        ExcelUtil<SysPlugin> util = new ExcelUtil<SysPlugin>(SysPlugin.class);
        util.importTemplateExcel(response, "用户数据");
    }

    /**
     * 根据插件编号获取详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:plugin:query')")
    @GetMapping(value = {"/", "/{pluginId}"})
    public AjaxResult getInfo(@PathVariable(value = "pluginId", required = false) Long pluginId) {

        AjaxResult ajax = AjaxResult.success();
        if (StringUtils.isNotNull(pluginId)) {
            SysPlugin sysPlugin = pluginService.selectPluginById(pluginId);
            ajax.put(AjaxResult.DATA_TAG, sysPlugin);
        }
        return ajax;
    }

    /**
     * 新增插件
     */
    @PreAuthorize("@ss.hasPermi('system:plugin:add')")
    @Log(title = "插件管理", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@Validated @RequestBody SysPlugin plugin) {
        if (UserConstants.NOT_UNIQUE.equals(pluginService.checkPluginNameUnique(plugin.getPluginName()))) {
            return AjaxResult.error("新增插件'" + plugin.getPluginName() + "'失败，插件信息已存在");
        }
        plugin.setCreateBy(getUsername());
        return toAjax(pluginService.insertPlugin(plugin));
    }

    /**
     * 修改插件
     */
    @PreAuthorize("@ss.hasPermi('system:plugin:edit')")
    @Log(title = "插件管理", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@Validated @RequestBody SysPlugin plugin) {
        plugin.setUpdateBy(getUsername());
        return toAjax(pluginService.updatePlugin(plugin));
    }

    /**
     * 删除插件
     */
    @PreAuthorize("@ss.hasPermi('system:plugin:remove')")
    @Log(title = "插件管理", businessType = BusinessType.DELETE)
    @DeleteMapping("/{pluginIds}")
    public AjaxResult remove(@PathVariable Long[] pluginIds) {
        return toAjax(pluginService.deletePluginByIds(pluginIds));
    }

    /**
     * 插件状态修改
     */
    @PreAuthorize("@ss.hasPermi('system:plugin:edit')")
    @Log(title = "插件管理", businessType = BusinessType.UPDATE)
    @PutMapping("/changeStatus")
    public AjaxResult changeStatus(@RequestBody SysPlugin plugin) {
        plugin.setUpdateBy(getUsername());
        return toAjax(pluginService.updatePluginStatus(plugin));
    }

    @PreAuthorize("@ss.hasPermi('system:plugin:uploadFile')")
    @Log(title = "插件上传", businessType = BusinessType.IMPORT)
    @PostMapping("/uploadFile")
    public AjaxResult uploadFile(MultipartFile file) {
        return AjaxResult.success(pluginService.uploadFile(file));
    }

    @PreAuthorize("@ss.hasPermi('system:plugin:getOptionByCategory')")
    @GetMapping("/getOptionByCategory/{categoryId}")
    public AjaxResult getOptionByCategory(@PathVariable Long categoryId) {
        return AjaxResult.success(pluginService.getOptionByCategory(categoryId));
    }
}
