package com.tbyf.his.web.dataImport.core;

import com.tbyf.his.common.utils.StringUtils;
import com.tbyf.his.web.dataImport.domain.vo.VerifyVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author lzz
 * @date 2023/2/9 9:22
 */
@Configuration
@Slf4j
public class DiConfig {

    @Autowired
    private ApplicationContext applicationContext;

    public static final Map<String, RuleValidator> VALIDATOR_MAP = new ConcurrentHashMap<>(8);

    private static final List<ValueConvert> CONVERT_LIST = new ArrayList<>();

    /**
     * 转换规则处理
     * @param vo
     * @return
     */
    public static String valueConvert(RuleVO vo){
        for (ValueConvert convert : CONVERT_LIST){
            if (convert.support(vo)){
                return convert.convert(vo);
            }
        }
        return vo.getValue();
    }

    /**
     * 获取对应列的数据起始位置
     *
     * @param vo
     * @param sheet
     * @return
     */
    public static String getCoordinate(VerifyVO vo, Sheet sheet) {
        final int rows = sheet.getLastRowNum() + 1;
        int start = StringUtils.isBlank(vo.getCode()) ? 1 : 0;
        for (int i = 0; i < rows; i++) {
            final Row row = sheet.getRow(i);
            int flag = start;
            for (Cell cell : row) {
                if (cell.getColumnIndex() == 0) {
                    break;
                }
                if (flag == 0 && StringUtils.equals(getValue(cell), vo.getCode())) {
                    flag++;
                } else if (flag == 1 && StringUtils.equals(getValue(cell), vo.getTitle())) {
                    // 代表code与title都能对应上
                    return getWord(cell.getColumnIndex() + 1) + "," + cell.getRowIndex() + 1;
                }
            }
        }
        return null;
    }

    /**
     * 获取横坐标数据结果,直接获取columnIndex
     *
     * @param vo
     * @param sheet
     * @return
     */
    public static List<Integer> getLength(VerifyVO vo, Sheet sheet) {
        final String[] coords = vo.getCoordinate().split(",");
        int start = DiConfig.getIndex(coords[0]);
        Row row = sheet.getRow(Integer.parseInt(coords[1]) - 1);
        List<Integer> list = new ArrayList<>();
        // 空值出现次数,出现次数大于2的时候停止循环
        int flag = 0;
        for (int i = start; flag <= 2; i++) {
            final Cell cell = row.getCell(i, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            final String value = getValue(cell);
            if (StringUtils.isBlank(value)) {
                flag++;
            } else if (StringUtils.equals(value, "-")) {
                flag = 0;
            } else {
                flag = 0;
                list.add(cell.getColumnIndex());
            }
        }
        return list;
    }

    @PostConstruct
    public void init() {
        final Map<String, RuleValidator> validatorMap = applicationContext.getBeansOfType(RuleValidator.class);
        VALIDATOR_MAP.putAll(validatorMap);
        Map<String, ValueConvert> convertMap = applicationContext.getBeansOfType(ValueConvert.class);
        CONVERT_LIST.addAll(convertMap.values());
    }

    /**
     * 获取校验器
     *
     * @param mode
     * @return
     */
    public static RuleValidator getValidator(String mode) {
        return VALIDATOR_MAP.get(DiConstants.SERVICE_PREFIX + mode);
    }

    /**
     * 将excel表格中列号字母转成列索引,从0对应A开始
     *
     * @param word
     * @return
     */
    public static int getIndex(String word) {
        if (!word.matches("[A-Z]+")) {
            return -1;
        }
        int index = 0;
        char[] chars = word.toUpperCase().toCharArray();
        for (int i = 0; i < chars.length; i++) {
            index += ((int) chars[i] - (int) 'A' + 1)
                    * (int) Math.pow(26, chars.length - i - 1);
        }
        return index - 1;
    }

    /**
     * 将excel表格中列索引转成列号字母,从A对应0开始
     *
     * @param index
     * @return
     */
    public static String getWord(int index) {
        if (index <= 0) {
            return "A";
        }
        String column = "";
        do {
            if (column.length() > 0) {
                index--;
            }
            column = ((char) (index % 26 + (int) 'A')) + column;
            index = (index - index % 26) / 26;
        } while (index > 0);
        return column;
    }

    /**
     * 获取单元格的值
     *
     * @param cell
     * @return
     */
    public static String getValue(Cell cell) {
        if (cell == null || "".equals(cell.toString().trim())) {
            return "";
        }
        String cellString = "";
        final CellType type = cell.getCellType();
        switch (type) {
            case STRING:
                cellString = trim(cell.getRichStringCellValue().getString());
                break;
            case BOOLEAN:
                cellString = String.valueOf(cell.getBooleanCellValue());
                break;
            case FORMULA:
                final FormulaEvaluator evaluator = cell.getSheet().getWorkbook().getCreationHelper().createFormulaEvaluator();
                cellString = evaluator.evaluate(cell).getStringValue().trim();
                break;
            case NUMERIC:
                final short format = cell.getCellStyle().getDataFormat();
                if (DateUtil.isCellDateFormatted(cell)) {
                    // 日期格式
                    SimpleDateFormat sdf = null;
                    if (format == 20 || format == 32) {
                        sdf = new SimpleDateFormat("HH:mm");
                        cellString = sdf.format(cell.getDateCellValue());
                    } else if (format == 14 || format == 31 || format == 57 || format == 58) {
                        // 处理自定义日期格式：m月d日(通过判断单元格的格式id解决，id的值是58)
                        sdf = new SimpleDateFormat("yyyy-MM-dd");
                        double value = cell.getNumericCellValue();
                        Date date = org.apache.poi.ss.usermodel.DateUtil
                                .getJavaDate(value);
                        cellString = sdf.format(date);
                    } else {// 日期
                        sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        cellString = sdf.format(cell.getDateCellValue());
                    }
                } else {
                    cellString = NumberToTextConverter.toText(cell.getNumericCellValue());
                }
                break;
            default:
                break;
        }
        return cellString;
    }

    public static String trim(String text) {
        return text.replaceAll("(\\u00A0+| )", "").trim();
    }

}
