package com.tbyf.his.adapter.service.impl;

import com.tbyf.his.adapter.domain.AdapterPublishColumn;
import com.tbyf.his.adapter.domain.AdapterPublishTable;
import com.tbyf.his.adapter.domain.AdapterPublishTask;
import com.tbyf.his.adapter.domain.PublishColumnDto;
import com.tbyf.his.adapter.mapper.AdapterPublishColumnMapper;
import com.tbyf.his.adapter.mapper.AdapterPublishTableMapper;
import com.tbyf.his.adapter.mapper.AdapterPublishTaskMapper;
import com.tbyf.his.adapter.service.AdapterPublishTaskService;
import com.tbyf.his.adapter.service.SqlGenerator;
import com.tbyf.his.adapter.task.StringColumnRowMapper;
import com.tbyf.his.adapter.util.AdapterConstant;
import com.tbyf.his.adapter.util.IncFieldType;
import com.tbyf.his.common.annotation.DataSource;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.enums.DataSourceType;
import com.tbyf.his.common.exception.ServiceException;
import com.tbyf.his.framework.datasource.DataSourceUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@Slf4j
@DataSource(value = DataSourceType.SLAVE)
public class AdapterPublishTaskServiceImpl implements AdapterPublishTaskService {

    @Autowired
    private Map<String, SqlGenerator> GENERATOR_MAP;

    @Autowired
    @Lazy
    private AdapterPublishTaskService adapterPublishTaskService;

    @Autowired
    private AdapterPublishTaskMapper adapterPublishTaskMapper;

    @Autowired
    private AdapterPublishTableMapper adapterPublishTableMapper;

    @Autowired
    private AdapterPublishColumnMapper adapterPublishColumnMapper;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Override
    public List<AdapterPublishTable> getList(String sourceName, String publishTableName) {
        return adapterPublishTaskMapper.getList(sourceName, publishTableName);
    }

    @Override
    public void startUp(List<Long> ids) {
        for (Long id : ids) {
            AdapterPublishTask adapterPublishTask = createTaskByTable(id);

            /**
             * 如果任务存在则更新任务
             * 如果任务不存在则新建任务
             */
            AdapterPublishTask task = adapterPublishTaskMapper.selectTaskByPublishTableNameAndSourceName(adapterPublishTask.getPublishTableName(), adapterPublishTask.getSourceName());
            if (!ObjectUtils.isEmpty(task)) {
                adapterPublishTask.setLastRecordTime(null);
                adapterPublishTaskMapper.updateAdapterPublishTask(adapterPublishTask);
            } else {
                adapterPublishTask.setTaskName(adapterPublishTask.getPublishTableName() + "_TASK");
                adapterPublishTaskMapper.insertAdapterPublishTask(adapterPublishTask);
            }

        }
    }

    @Override
    public void stop(List<Long> ids) {
        for (Long id : ids) {
            AdapterPublishTable adapterPublishTable = adapterPublishTableMapper.selectByPrimaryKey(id);
            String sourceName = adapterPublishTable.getSourceName();
            String publishTableName = adapterPublishTable.getPublishTableName();

            /**
             * 如果任务存在则停止任务
             */
            AdapterPublishTask task = adapterPublishTaskMapper.selectTaskByPublishTableNameAndSourceName(publishTableName, sourceName);
            if (!ObjectUtils.isEmpty(task)) {
                task.setStatus(AdapterConstant.TASK_STATUS_DISABLE);
                adapterPublishTaskMapper.updateAdapterPublishTaskStatus(task.getStatus(), task.getId());
            }
        }
    }

    @Override
    public AdapterPublishTask selectTaskByPublishTableNameAndSourceName(String publishTableName, String sourceName) {
        return adapterPublishTaskMapper.selectTaskByPublishTableNameAndSourceName(publishTableName, sourceName);
    }

    @Override
    public int updateAdapterPublishTask(AdapterPublishTask adapterPublishTask) {
        return adapterPublishTaskMapper.updateTaskInfo(adapterPublishTask);
    }

    @Override
    public List<AdapterPublishTask> selectTaskByEnable() {
        return adapterPublishTaskMapper.selectTaskByStatus("0");
    }

    @Override
    public List<Map<String, String>> queryIncList(AdapterPublishTask task) {
        try {
            DataSourceUtil.switchDs(task.getSourceName());
            final List<Map<String, String>> list = jdbcTemplate.query(StrFormatter.format(task.getIncSql(), task.getLastRecordTime()), new StringColumnRowMapper());
            //final List<Map<String, Object>> mapList = jdbcTemplate.queryForList(task.getIncSql());
            return CollectionUtils.isEmpty(list) ? Collections.emptyList() : list;
        } finally {
            DataSourceUtil.switchDefaultDs();
        }
    }

    @Override
    public List<PublishColumnDto> selectColumnByTableSource(String tableName, String dataSourceId) {
        return adapterPublishTaskMapper.selectColumnByTableSource(tableName, dataSourceId);
    }

    @Override
    public void generatorSql(AdapterPublishTask task, String data) {
        final String sourceType = adapterPublishTaskService.getDataSourceType(task.getSourceName());
        final SqlGenerator sqlGenerator = GENERATOR_MAP.get(AdapterConstant.GENERATOR_PREFIX + sourceType);
        assert sqlGenerator != null;
        task.setLastRecordTime(data);
        final String sql = sqlGenerator.generator(task);
        final AdapterPublishTask updateTask = new AdapterPublishTask();
        updateTask.setId(task.getId()).setLastRecordTime(String.valueOf(data)).setIncSql(sql);
        DataSourceUtil.switchDs(DataSourceType.SLAVE.name());
        adapterPublishTaskMapper.updateTaskInfo(updateTask);
        log.info("更新增量sql:[{}:{}],data:[{}]", task.getId(), task.getTaskName(), data);
    }

    @Override
    public String getDataSourceType(String dataSourceId) {
        final String driver = adapterPublishTaskMapper.getDataSourceDriver(dataSourceId);
        if (StringUtils.hasText(driver)) {
            if (driver.toLowerCase().contains(AdapterConstant.MYSQL)) {
                return AdapterConstant.MYSQL;
            } else if (driver.toLowerCase().contains(AdapterConstant.ORACLE)) {
                return AdapterConstant.ORACLE;
            }
        }
        return AdapterConstant.MYSQL;
    }

    private AdapterPublishTask createTaskByTable(Long id) {
        AdapterPublishTable adapterPublishTable = adapterPublishTableMapper.selectByPrimaryKey(id);
        String sourceTableName = adapterPublishTable.getSourceTableName();
        String incField = adapterPublishTable.getIncField();
        String sourceName = adapterPublishTable.getSourceName();
        String publishTableName = adapterPublishTable.getPublishTableName();

        String dbType = getDbType(sourceName);

        /**
         * 校验是否发布了发布表表对应数据集的主数据元列
         */
        String primaryMetadataCode = adapterPublishColumnMapper.getPrimaryMetadataCode(adapterPublishTable.getDatasetCode());
        List<String> metadataCodeList = adapterPublishColumnMapper.getMetadataCodeList(adapterPublishTable.getSourceTableName(), adapterPublishTable.getSourceName());
        if (!metadataCodeList.contains(primaryMetadataCode)) {
            throw new ServiceException("发布表" + adapterPublishTable.getPublishTableName() + "未发布数据集" + adapterPublishTable.getDatasetCode() + "的数据元主键" + primaryMetadataCode);
        }

        /**
         * 根据所有的发布列生成，查询字段和关联查询条件
         */
        List<AdapterPublishColumn> columnList = adapterPublishColumnMapper.getColumnList(sourceTableName, sourceName);
        String columns = getSelectColumn(columnList, incField, sourceTableName);
        String relationConditions = getRelationConditions(columnList);

        DataSourceUtil.switchDs(sourceName);
        String sql = "";
        if ("oracle".equals(dbType)) {
            sql = StrFormatter.format("SELECT {} FROM (SELECT {} FROM {} ORDER BY {}) WHERE ROWNUM = 1",
                    incField, incField, sourceTableName, incField);
        } else if ("mysql".equals(dbType)) {
            sql = StrFormatter.format("SELECT X.* FROM (SELECT {} FROM {} ORDER BY {}) X LIMIT 0,1",
                    incField, sourceTableName, incField);
        }
        String lastRecordTime = jdbcTemplate.queryForObject(sql, String.class);
        DataSourceUtil.switchDs(DataSourceType.SLAVE.name());

        String incSql = "";
        if (IncFieldType.DATE.equals(adapterPublishTable.getIncFieldType())) {
            if ("oracle".equals(dbType)) {
                incSql = StrFormatter.format("SELECT X.* FROM (SELECT {} FROM {} {} WHERE {}.{} > TO_DATE('\\{}','yyyy-mm-dd hh24:mi:ss') ORDER BY {}.{}) X WHERE ROWNUM <= 500",
                        columns, sourceTableName, relationConditions, sourceTableName, incField, sourceTableName, incField);
            } else if ("mysql".equals(dbType)) {
                incSql = StrFormatter.format("SELECT X.* FROM (SELECT {} FROM {} {} WHERE {} > str_to_date('\\{}','%Y-%m-%d %T') ORDER BY {}.{}) X LIMIT 0,500",
                        columns, sourceTableName, relationConditions, sourceTableName, incField, sourceTableName, incField);
            }
        } else {
            if ("oracle".equals(dbType)) {
                incSql = StrFormatter.format("SELECT X.* FROM (SELECT {} FROM {} {} WHERE {}.{} > \\{} ORDER BY {}.{}) X WHERE ROWNUM <= 500",
                        columns, sourceTableName, relationConditions, sourceTableName, incField, sourceTableName, incField);
            } else if ("mysql".equals(dbType)) {
                incSql = StrFormatter.format("SELECT X.* FROM (SELECT {} FROM {} {} WHERE {}.{} > \\{} ORDER BY {}.{}) X LIMIT 0,500",
                        columns, sourceTableName, relationConditions, sourceTableName, incField, sourceTableName, incField);
            }
        }

        System.out.println(incSql);

        return new AdapterPublishTask()
                .setPublishTableName(publishTableName)
                .setIncField(incField)
                .setIncFieldType(adapterPublishTable.getIncFieldType())
                .setIncSql(incSql)
                .setLastRecordTime(lastRecordTime)
                .setStatus(AdapterConstant.TASK_STATUS_ENABLE)
                .setSourceName(sourceName);
    }

    private String getSelectColumn(List<AdapterPublishColumn> columnList, String incField, String sourceTableName) {

        StringBuilder stringBuilder = new StringBuilder();

        columnList.forEach(column -> {
            switch (column.getConvertMode()) {
                //普通列
                case AdapterConstant.COMMON_COLUMN:
                    //值域转换
                case AdapterConstant.RANGE_CONVERT:
                    stringBuilder.append(column.getSourceTableName()).append(".").append(column.getColumnName()).append(",");
                    break;
                //公式转换
                case AdapterConstant.FORMULA_CONVERT:
                    //SQL转换
                case AdapterConstant.SQL_CONVERT:
                    //常量转换
                case AdapterConstant.CONSTANT_CONVERT:
                    stringBuilder
                            .append(column.getColumnConvertValue()).append(" AS ")
                            .append(column.getSourceTableName()).append(".").append(column.getColumnName()).append(",");
                    break;
                //基础数据转换
                case AdapterConstant.BASE_DATA:
                    stringBuilder.append(column.getRelationQueryTable()).append(".").append(column.getRelationResultColumn()).append(" AS ").append(column.getColumnName()).append(",");
                    break;
                default:
                    log.error("发布列转换类型异常");
            }
        });

        //如果未发布增量字段则加上增量字段
        if (columnList.stream().noneMatch(column -> incField.equals(column.getColumnName()))) {
            stringBuilder.append(sourceTableName).append(".").append(incField).append(",");
        }

        return stringBuilder.substring(0, stringBuilder.length() - 1);
    }

    private String getRelationConditions(List<AdapterPublishColumn> columnList) {

        columnList = columnList.stream().filter(column -> AdapterConstant.BASE_DATA.equals(column.getConvertMode())).collect(Collectors.toList());

        if (columnList.size() == 0) {
            return "";
        } else {
            StringBuilder stringBuilder = new StringBuilder();
            columnList.forEach(column -> {
                stringBuilder
                        .append("LEFT JOIN ")
                        .append(column.getRelationQueryTable())
                        .append(" ON ")
                        .append(column.getSourceTableName()).append(".").append(column.getSourceRelationColumn())
                        .append(" = ")
                        .append(column.getRelationQueryTable()).append(".").append(column.getRelationQueryColumn())
                        .append(" ");
            });
            return stringBuilder.substring(0, stringBuilder.length() - 1);
        }
    }

    private String getDbType(String sourceName) {
        DataSourceUtil.switchDs(DataSourceType.MASTER.name());
        String selectDbDriverClass = String.format("SELECT driver_class FROM `sys_datasource` WHERE datasource_name = '%s'", sourceName);
        String dbDriverClass = jdbcTemplate.queryForObject(selectDbDriverClass, String.class);
        DataSourceUtil.switchDs(DataSourceType.SLAVE.name());

        if (dbDriverClass != null && dbDriverClass.toLowerCase().contains("mysql")) {
            return "mysql";
        } else if (dbDriverClass != null && dbDriverClass.toLowerCase().contains("oracle")) {
            return "oracle";
        } else {
            throw new ServiceException("数据源驱动异常");
        }
    }
}
