package com.tbyf.his.adapter.task;

import com.tbyf.his.adapter.domain.AdapterPublishTask;
import com.tbyf.his.adapter.domain.PublishColumnDto;
import com.tbyf.his.adapter.mapper.AdapterPublishTaskMapper;
import com.tbyf.his.adapter.service.AdapterPublishTaskService;
import com.tbyf.his.common.core.text.StrFormatter;
import com.tbyf.his.common.utils.DateUtils;
import com.tbyf.his.common.utils.uuid.IdUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 适配器调度任务
 *
 * @author lzz
 * @date 2022/12/9 9:27
 */
@Component("adapterTask")
@Slf4j
public class AdapterTask {

    @Autowired
    private AdapterPublishTaskService adapterPublishTaskService;

    /**
     * 使用springboot内置线程池,未自定义
     */
    @Autowired
    private TaskExecutor taskExecutor;

    @Autowired
    private AdapterPublishTaskMapper adapterPublishTaskMapper;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private AdapterProperties adapterProperties;


    @Autowired
    private DataAdapterService dataAdapterService;

    /**
     * 存储任务标识与其执行情况
     */
    public static final Map<String, AtomicBoolean> TASK_EXECUTE_MAP = new ConcurrentHashMap<>();

    public static final String SOAP_TEMPLATE = "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:web=\"http://webservice.project.web.platfrom.hip.tbyf.com/\">\n" +
            "<soapenv:Header/><soapenv:Body><web:invoke><arg0><![CDATA[{}]]></arg0></web:invoke></soapenv:Body></soapenv:Envelope>";

    /**
     * 获取执行任务ID的执行情况
     *
     * @param taskId
     * @return
     */
    public static boolean getTaskExecution(String taskId) {
        final AtomicBoolean aBoolean = TASK_EXECUTE_MAP.get(taskId);
        if (aBoolean != null) {
            return aBoolean.get();
        }
        return false;
    }

    /**
     * 获取或初始化任务执行情况
     * 初始化值为false
     *
     * @param taskId
     * @return
     */
    private AtomicBoolean getOrInitExecute(String taskId) {
        return TASK_EXECUTE_MAP.computeIfAbsent(taskId, k -> new AtomicBoolean(false));
    }

    /**
     * 设置任务执行情况
     */
    private AtomicBoolean setTaskExecute(String taskId, boolean flag) {
        final AtomicBoolean atomicBoolean = getOrInitExecute(taskId);
        atomicBoolean.set(flag);
        return atomicBoolean;
    }


    /**
     * 执行适配器任务
     */
    public void executeAdapterTask() {
        final String trackId = IdUtils.fastSimpleUUID();
        log.info("==========[ {} Adapter调度任务开启 ]==========", trackId);
        final List<AdapterPublishTask> taskList = adapterPublishTaskService.selectTaskByEnable();
        List<AdapterPublishTask> runningList = new ArrayList<>();
        for (AdapterPublishTask task : taskList) {
            if (!getTaskExecution(String.valueOf(task.getId()))) {
                log.info("==========[ {} Adapter可调度任务:[{}] ]==========", trackId, task.getTaskName());
                runningList.add(task);
            }
        }
        log.info("==========[ {} Adapter本次调度任务总数量:[{}] ]==========", trackId, runningList.size());
        for (AdapterPublishTask task : runningList) {
            taskExecutor.execute(() -> {
                setTaskExecute(String.valueOf(task.getId()), true);
                final long startTime = System.currentTimeMillis();
                // 执行成功数量
                int successCount = 0;
                // 源数据集合
                List<Map<String, String>> sourceList = null;
                // 代理数据集合
                List<String> targetList = null;
                try {
                    sourceList = adapterPublishTaskService.queryIncList(task);
                    log.info("==========[ {} Adapter任务:[{}]增量数据数量:[{}] ]==========", trackId, task.getTaskName(), sourceList.size());
                    if (CollectionUtils.isEmpty(sourceList)) {
                        return;
                    }
                    final List<PublishColumnDto> columnList = adapterPublishTaskService.selectColumnByTableSource(task.getPublishTableName(), task.getSourceName());
                    if (CollectionUtils.isEmpty(columnList)) {
                        log.info("==========[ {} Adapter任务:[{}]发布列为空,请检查配置 ]==========", trackId, task.getTaskName());
                        return;
                    }
                    // 源数据转换成xml代理数据
                    targetList = dataAdapterService.columnConvertByXml(sourceList, columnList);
                    for (; successCount < targetList.size(); successCount++) {
                        final String result = DataAdapterService.doSoap(adapterProperties.getSyncPath(), StrFormatter.format(SOAP_TEMPLATE, targetList.get(successCount)));
                        log.debug("==========[ {} Adapter任务[{}]发布返回值:[{}] ]==========", trackId, task.getTaskName(), result);
                        // 校验返回值判断本次发布是否成功
                        DataAdapterService.verifyXmlResult(result);
                    }
                } catch (Exception e) {
                    log.error(StrFormatter.format("==========[ {} Adapter任务[{}] 异常中断]==========", trackId, task.getTaskName()), e);
                    adapterPublishTaskMapper.updateTaskInfo(new AdapterPublishTask().setId(task.getId()).setErrorInfo(DateUtils.getTime() + e.getMessage()));
                } finally {
                    log.info("==========[ {} Adapter任务:[{}] 适配成功数量:[{}] 耗时:[{}]ms ]==========", trackId, task.getTaskName(), successCount, System.currentTimeMillis() - startTime);
                    if (successCount > 0 && !CollectionUtils.isEmpty(sourceList)) {
                        String lastData = sourceList.get(successCount - 1).get(task.getIncField());
                        adapterPublishTaskMapper.updateTaskInfo(new AdapterPublishTask().setId(task.getId()).setLastRecordTime(lastData));
                    }
                    setTaskExecute(String.valueOf(task.getId()), false);
                }
            });
        }
    }

    @SneakyThrows
    @PostConstruct
    public void init() {
        //http://192.168.0.153:15001/datacenter/rpc/webservice/SubscriptionService?wsdl
        //dataAdapterService.doSoap("http://192.168.0.153:15001/datacenter/rpc/webservice/SyncMessageService?wsdl","")
        //executeAdapterTask();
    }
}
