package com.tbyf.his.tool.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tbyf.his.common.annotation.DataScope;
import com.tbyf.his.common.core.domain.entity.SysMenu;
import com.tbyf.his.common.core.page.TableDataInfo;
import com.tbyf.his.common.exception.ServiceException;
import com.tbyf.his.common.utils.DateUtils;
import com.tbyf.his.framework.system.mapper.SysRoleDeptMapper;
import com.tbyf.his.framework.system.service.ISysMenuService;
import com.tbyf.his.system.mapper.SysDynamicFormMenuMapper;
import com.tbyf.his.system.service.ISysDatasourceService;
import com.tbyf.his.system.vo.DynamicSqlData;
import com.tbyf.his.system.vo.SqlHandler;
import com.tbyf.his.system.vo.SysMenuVo;
import com.tbyf.his.tool.domain.DynamicForm;
import com.tbyf.his.tool.mapper.DynamicFormMapper;
import com.tbyf.his.tool.service.IDynamicFormService;
import com.tbyf.his.tool.vo.*;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author: fr
 * @date: 2022年08月17日 11:37
 */
@Service
public class DynamicFormServiceImpl implements IDynamicFormService {
    @Resource
    private DynamicFormMapper dynamicFormMapper;

    @Autowired
    private ISysDatasourceService sysDatasourceService;

    @Autowired
    private ISysMenuService sysMenuService;

    @Resource
    SysDynamicFormMenuMapper sysDynamicFormMenuMapper;

    /**
     * 查询动态单
     *
     * @param formId 动态单主键
     * @return 动态单
     */
    @Override
    public DynamicForm selectDynamicFormByFormId(Long formId) {
        return dynamicFormMapper.selectDynamicFormByFormId(formId);
    }

    /**
     * 查询动态单列表
     *
     * @param dynamicForm 动态单
     * @return 动态单
     */
    @Override
    @DataScope(
            deptAlias = "d",
            userAlias = "u"
    )
    public List<DynamicForm> selectDynamicFormList(DynamicForm dynamicForm) {

        return dynamicFormMapper.selectDynamicFormList(dynamicForm);
    }

    /**
     * 新增动态单
     *
     * @param dynamicForm 动态单
     * @return 结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int insertDynamicForm(DynamicForm dynamicForm) {
        dynamicForm.setCreateTime(DateUtils.getNowDate());
        //新增表单
//        List<SysMenu> sysMenuPermission = getSysMenuPermission(dynamicForm.getFormId(), dynamicForm.getCreateBy(),dynamicForm.getFormName());
//        sysMenuPermission.forEach(e->{
//            sysMenuService.insertMenu(e);
//        });
        return dynamicFormMapper.insertDynamicForm(dynamicForm);
    }

    /**
     * 修改动态单
     *
     * @param dynamicForm 动态单
     * @return 结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int updateDynamicForm(DynamicForm dynamicForm) {
        // 先看数据库是否存在
        Integer existTable = dynamicFormMapper.existTable(dynamicForm.getDatasourceId(), dynamicForm.getTableName(), dynamicForm.getFormId());
        if (null != existTable) {
            throw new ServiceException("该表已被其他表单绑定，请不要重复添加");
        }
        Integer have = dynamicFormMapper.isHave(dynamicForm.getDatasourceId(), dynamicForm.getTableName());
        //生成菜单
        if (null == have) {
            SysMenuVo menuVo = SysMenuVo.builder().menuName(dynamicForm.getFormName()).parentId(3l)
                    .orderNum(dynamicForm.getFormId().intValue())
                    .isFrame("1").isCache("0").visible("0")
                    .menuType("C").status("0").icon("form")
                    .perms(dynamicForm.getFormId() + ":list").applicationId(7l)
                    .component("tool/dynamicForm/data").path("dynamicFrom/data/index").query("{" + "\"" + "id" + "\"" + ":" + dynamicForm.getFormId() + "}").build();
            menuVo.setCreateBy(dynamicForm.getCreateBy());
            SysMenu formMenu = new SysMenu();
            BeanUtils.copyProperties(menuVo, formMenu);
            sysDynamicFormMenuMapper.insertMenu(formMenu);
            List<SysMenu> sysMenuPermission = getSysMenuPermission(dynamicForm.getFormId(), formMenu);
            sysMenuPermission.forEach(e -> {
                sysMenuService.insertMenu(e);
            });
        }
        dynamicForm.setUpdateTime(DateUtils.getNowDate());
        return dynamicFormMapper.updateDynamicForm(dynamicForm);
    }

    /**
     * 批量删除动态单
     *
     * @param formId 需要删除的动态单主键
     * @return 结果
     */
    @Override
    public int deleteDynamicFormByFormIds(Long formId) {
        dynamicFormMapper.deleteLogic(formId);
        return 0;
    }

    /**
     * 删除动态单信息
     *
     * @param formId       动态单主键
     * @param datasourceId 数据源id
     * @param tableName    表名
     * @return 结果
     */
    @Override
    public int deleteDynamicFormByFormId(Long formId, Long datasourceId, String tableName) {
        // 清空表数据
        if (StringUtils.hasText(tableName)) {
            sysDatasourceService.executeDDL(datasourceId, String.format(SqlHandler.DELETE_TABLE_ALL_DATA, tableName));
        }
        return dynamicFormMapper.deleteDynamicFormByFormId(formId);
    }

    @Override
//    @DataScope(deptAlias = "d") // dataScope ->  AND (d.dept_id = 100 )
    public TableDataVO dynamicList(TableDataSearch search) {
        System.out.println(search.getParams());
        // 先查询表单对应数据
        DynamicForm form = selectDynamicFormByFormId(search.getPid());
        if (StringUtils.isEmpty(form.getFormContent())) {
            throw new ServiceException("请先设计表单，再查看数据");
        }
        // 获取拼接sql需要的一些基本参数
        FormContent formContent = JSON.parseObject(form.getFormContent(), FormContent.class);
        // 主键要加入到查询条件中
        String primaryKey = String.valueOf(formContent.getFormConf().get("primaryKey"));
        List<String> queryColumn = formContent.getDrawingList().stream().map(d -> String.valueOf(d.get("vModel"))).collect(Collectors.toList());
        queryColumn.add(0, primaryKey);
        TableDataInfo info = sysDatasourceService.selectPageByParam(DynamicSqlData.builder()
                .datasourceId(form.getDatasourceId())
                .tableName(form.getTableName())
                .queryParam(queryColumn)
                .whereParam(search.getSearchParam())
                .pageNum(search.getPageNum())
                .pageSize(search.getPageSize()).build());
        // 组装成页面数据格式
        List<KeyData> keyData = formContent.getDrawingList().stream().map(d -> KeyData.builder()
                .field(String.valueOf(d.get("vModel")))
                .label(String.valueOf(d.get("label"))).build()).collect(Collectors.toList());
        List<Map<String, Object>> rows = (List<Map<String, Object>>) info.getRows();
        rows.forEach(row -> {
            Object obj = row.get(SqlHandler.ORACLE_PRIMARY_KEY_ALIAS);
            if (null != obj) {
                RowId rowId = JSONObject.parseObject(JSON.toJSONString(obj), RowId.class);
                // 会有byte数组序列化问题，fastjson会有base64加密处理
                String data = RowId.getData(rowId.getBytes());
                row.put(SqlHandler.ORACLE_PRIMARY_KEY_ALIAS, data);
            }
        });
        rows.forEach(d -> d.put("keyData", keyData));
        return TableDataVO.builder()
                .pid(form.getFormId())
                .primaryKey(primaryKey)
                .total(info.getTotal())
                .tableName(form.getTableName())
                .datasourceId(form.getDatasourceId())
                .drawDataOne(formContent)
                .tableList(rows).build();
    }

    @Override
    public void dynamicAdd(DynamicTableSave data) {
        sysDatasourceService.saveByParam(DynamicSqlData.builder()
                .datasourceId(data.getDatasourceId())
                .tableName(data.getTableName())
                .primaryKey(data.getPrimaryKey())
                .dataParam(data.getDataParam()).build());
    }

    @Override
    public void dynamicEdit(DynamicTableSave data) {
        sysDatasourceService.saveByParam(DynamicSqlData.builder()
                .datasourceId(data.getDatasourceId())
                .tableName(data.getTableName())
                .primaryParam(data.getPrimaryParam())
                .primaryKey(data.getPrimaryKey())
                .dataParam(data.getDataParam()).build());
    }

    @Override
    public void dynamicImport(List<DynamicTableSave> list, Boolean isCover) {
        if (CollectionUtils.isEmpty(list)) {
            throw new ServiceException("数据为空");
        }
        sysDatasourceService.saveBatchByParam(list.stream()
                .map(data -> DynamicSqlData.builder()
                        .datasourceId(data.getDatasourceId())
                        .tableName(data.getTableName())
                        .primaryParam(data.getPrimaryParam())
                        .primaryKey(data.getPrimaryKey())
                        .dataParam(data.getDataParam()).build())
                .collect(Collectors.toList()), isCover);
    }

    @Override
    public void dynamicDelete(DynamicTableSave data) {
        sysDatasourceService.deleteByParam(DynamicSqlData.builder()
                .datasourceId(data.getDatasourceId())
                .tableName(data.getTableName())
                .primaryParam(data.getPrimaryParam())
                .primaryKey(data.getPrimaryKey())
                .dataParam(data.getDataParam()).build());
    }

    public List<SysMenu> getSysMenuPermission(Long formId, SysMenu menuVo) {
        //权限菜单列表
        List<SysMenu> sysMenus = new ArrayList<>();
        menuVo.setComponent(null);
        menuVo.setPath(null);
        menuVo.setMenuType("F");
        menuVo.setIcon("#");
        menuVo.setParentId(menuVo.getMenuId());
        menuVo.setMenuId(null);
        menuVo.setQuery(null);
        //数据新增
        SysMenu dataAdd = new SysMenu();
        BeanUtils.copyProperties(menuVo, dataAdd);
        dataAdd.setMenuName("新增");
        dataAdd.setPerms(formId + ":add");
        sysMenus.add(dataAdd);
        //数据编辑
        SysMenu dataEdit = new SysMenu();
        BeanUtils.copyProperties(menuVo, dataEdit);
        dataEdit.setMenuName("编辑");
        dataEdit.setPerms(formId + ":edit");
        sysMenus.add(dataEdit);
        //数据删除
        SysMenu dataDelete = new SysMenu();
        BeanUtils.copyProperties(menuVo, dataDelete);
        dataDelete.setMenuName("删除");
        dataDelete.setPerms(formId + ":delete");
        sysMenus.add(dataDelete);
        //数据导入
        SysMenu dataImport = new SysMenu();
        BeanUtils.copyProperties(menuVo, dataImport);
        dataImport.setMenuName("导入");
        dataImport.setPerms(formId + ":import");
        sysMenus.add(dataImport);
        //数据导出
        SysMenu dataExport = new SysMenu();
        BeanUtils.copyProperties(menuVo, dataExport);
        dataExport.setMenuName("导出");
        dataExport.setPerms(formId + ":export");
        sysMenus.add(dataExport);
        return sysMenus;
    }
}
