import request from '@/utils/request'

const prefix = '/dataImport'

/**
 * 模板查询
 * @param params
 * @returns {*}
 */
export function queryTemplate(params) {
    return request({
        url: `${prefix}/template`,
        method: 'get',
        params
    })
}

/**
 * 新增模板
 * @param data
 * @returns {*}
 */
export function addTemplate(data) {
    return request({
        url: `${prefix}/template`,
        method: 'post',
        data
    })
}

/**
 * 修改模板
 * @param data
 * @returns {*}
 */
export function updateTemplate(data) {
    return request({
        url: `${prefix}/template`,
        method: 'put',
        data
    })
}

/**
 * 删除模板
 * @param templateId
 * @returns {*}
 */
export function deleteTemplate(templateId) {
    return request({
        url: `${prefix}/template`,
        method: 'delete',
        params:{templateId}
    })
}


/**
 * 字典查询
 * @param type
 * @returns {*}
 */
export function queryDict(type) {
    return request({
        url: `${prefix}/dict`,
        method: 'get',
        params:{type}
    })
}

/**
 * 新增字典
 * @param data
 * @returns {*}
 */
export function addDict(data) {
    return request({
        url: `${prefix}/dict`,
        method: 'post',
        data
    })
}

/**
 * 修改字典
 * @param data
 * @returns {*}
 */
export function updateDict(data) {
    return request({
        url: `${prefix}/dict`,
        method: 'put',
        data
    })
}

/**
 * 删除字典
 * @param dictId
 * @returns {*}
 */
export function deleteDict(dictId) {
    return request({
        url: `${prefix}/dict`,
        method: 'delete',
        params:{dictId}
    })
}

/**
 * 查询数据源字典
 * @returns {AxiosPromise<any>}
 */
export function queryDatasource() {
    return request({
        url: `${prefix}/datasource`,
        method: 'get'
    })
}

/**
 * 查询表信息
 * @param dataSourceId
 * @returns {AxiosPromise<any>}
 */
export function queryTable(dataSourceId) {
    return request({
        url: `${prefix}/datasource/table`,
        method: 'get',
        params:{dataSourceId}
    })
}

/**
 * 查询数据库字段字典
 * @param dataSourceId
 * @param tableName
 * @returns {AxiosPromise}
 */
export function queryDbField(dataSourceId,tableName) {
    return request({
        url: `${prefix}/datasource/field`,
        method: 'get',
        params:{dataSourceId,tableName}
    })
}


/**
 * 字段查询
 * @param params
 * @returns {*}
 */
export function queryField(params) {
    return request({
        url: `${prefix}/field`,
        method: 'get',
        params
    })
}

/**
 * 新增字段
 * @param data
 * @returns {*}
 */
export function addField(data) {
    return request({
        url: `${prefix}/field`,
        method: 'post',
        data
    })
}

/**
 * 修改字段
 * @param data
 * @returns {*}
 */
export function updateField(data) {
    return request({
        url: `${prefix}/field`,
        method: 'put',
        data
    })
}

/**
 * 删除字段
 * @param fieldId
 * @returns {*}
 */
export function deleteField(fieldId) {
    return request({
        url: `${prefix}/field`,
        method: 'delete',
        params:{fieldId}
    })
}


/**
 * 规则查询
 * @param params
 * @returns {*}
 */
export function queryRule(params) {
    return request({
        url: `${prefix}/rule`,
        method: 'get',
        params
    })
}

/**
 * 规则字典
 * @returns {*}
 */
export function queryRuleDict() {
    return request({
        url: `${prefix}/rule/dict`,
        method: 'get'
    })
}

/**
 * 新增规则
 * @param data
 * @returns {*}
 */
export function addRule(data) {
    return request({
        url: `${prefix}/rule`,
        method: 'post',
        data
    })
}

/**
 * 修改规则
 * @param data
 * @returns {*}
 */
export function updateRule(data) {
    return request({
        url: `${prefix}/rule`,
        method: 'put',
        data
    })
}

/**
 * 删除规则
 * @param ruleId
 * @returns {*}
 */
export function deleteRule(ruleId) {
    return request({
        url: `${prefix}/rule`,
        method: 'delete',
        params:{ruleId}
    })
}

/**
 * 规则测试
 * @param data
 * @returns {*}
 */
export function ruleTest(data) {
    return request({
        url: `${prefix}/rule/test`,
        method: 'post',
        data
    })
}

/**
 * 快速绑定
 * @param templateId
 * @param ruleId
 * @returns {AxiosPromise}
 */
export function quickBind(templateId,ruleId) {
    return request({
        url: `${prefix}/bind/quick`,
        method: 'get',
        params:{templateId,ruleId}
    })
}

/**
 * 查询绑定的规则
 * @param fieldId
 * @returns {AxiosPromise}
 */
export function queryBindRule(fieldId) {
    return request({
        url: `${prefix}/bind`,
        method: 'get',
        params:{fieldId}
    })
}

/**
 * 配置字段规则绑定
 * @param data
 * @returns {AxiosPromise}
 */
export function bindRule(data) {
    return request({
        url: `${prefix}/bind`,
        method: 'post',
        data
    })
}

/**
 * 上传excel
 * @param data
 * @returns {*}
 */
export function uploadExcel(data) {
    return request({
        url: `${prefix}/upload`,
        method: 'post',
        data,
        headers: {
            'Content-Type': 'multipart/form-data'
        }
    })
}

/**
 * 下载模板
 * @param excelId
 * @returns {*}
 */
export function downloadTemplate(excelId) {
    return request({
        url: `${prefix}/template/download`,
        method: 'get',
        params:{excelId},
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
        responseType: "blob",
    })
}

/**
 * 下载模板
 * @param excelId
 * @returns {*}
 */
export function getTemplateFileName(excelId) {
    return request({
        url: `${prefix}/template/download/filename`,
        method: 'get',
        params:{excelId}
    })
}

export async function blobValidate(data) {
    try {
        const text = await data.text();
        JSON.parse(text);
        return false;
    } catch (error) {
        return true;
    }
}

/**
 * 根据基础模板重置字段
 * @param excelId
 * @returns {*}
 */
export function resetField(excelId){
    return request({
        url: `${prefix}/field/reset`,
        method: 'get',
        params:{excelId}
    })
}

/**
 * 创建物理表
 * @param tableName
 * @param templateId
 * @returns {*}
 */
export function createTable(tableName,templateId){
    return request({
        url: `${prefix}/table/create`,
        method: 'get',
        params:{tableName,templateId}
    })
}

/**
 * 数据分析并导出
 * @param templateId
 * @returns {*}
 */
export function analyzeExport(templateId){
    return request({
        url: `${prefix}/analyze/export`,
        method: 'get',
        params:{templateId},
        responseType: "blob",
    })
}

/**
 * 清空数据
 * @param data
 * @returns {*}
 */
export function clearTemp(data){
    return request({
        url: `${prefix}/clearTemp`,
        method: 'post',
        data
    })
}

/**
 * 一键导入正式库
 * @param data
 * @returns {*}
 */
export function syncDb(data){
    return request({
        url: `${prefix}/syncDb`,
        method: 'post',
        data
    })
}


/**
 * 查询模板的所有字段数据
 * @param templateId
 * @returns {*}
 */
export function getAllTemplateField(templateId) {
    return request({
        url: `${prefix}/template/allField`,
        method: 'get',
        params:{templateId}
    })
}

/**
 * 查询模板的所有规则数据
 * @param templateId
 * @returns {*}
 */
export function getAllTemplateRule(templateId) {
    return request({
        url: `${prefix}/template/allRule`,
        method: 'get',
        params:{templateId}
    })
}


/**
 * 新增规则
 * @param data
 * @returns {*}
 */
export function addRuleByTemplate(data) {
    return request({
        url: `${prefix}/rule/template`,
        method: 'post',
        data
    })
}

/**
 * 修改规则
 * @param data
 * @returns {*}
 */
export function updateRuleByTemplate(data) {
    return request({
        url: `${prefix}/rule/template`,
        method: 'put',
        data
    })
}

/**
 * 删除规则
 * @param ruleId
 * @param templateId
 * @returns {*}
 */
export function deleteRuleByTemplate(ruleId,templateId) {
    return request({
        url: `${prefix}/rule/template`,
        method: 'delete',
        params:{ruleId,templateId}
    })
}
