import axios from 'axios'
import { Loading, Message, MessageBox, Notification } from 'element-ui'
import store from '@/store'
import { getToken, removeToken } from '@/utils/auth'
import errorCode from '@/utils/errorCode'
import { blobValidate, tansParams } from '@/utils/ruoyi'
import cache from '@/plugins/cache'
import { saveAs } from 'file-saver'

let downloadLoadingInstance
// 是否显示重新登录
let isReloginShow

axios.defaults.headers['Content-Type'] = 'application/json;charset=utf-8'
const defaultSettings = require('@/settings')

// 创建axios实例
const service = axios.create({
  // axios中请求配置有baseURL选项，表示请求URL公共部分
  baseURL: process.env.VUE_APP_BASE_API,
  // 超时
  timeout: 1000*60*30
})

// request拦截器
service.interceptors.request.use(
  (config) => {
    // 是否需要设置 token
    const isToken = (config.headers || {}).isToken === false
    // 是否需要防止数据重复提交
    const isRepeatSubmit = (config.headers || {}).repeatSubmit === false
    if (getToken() && !isToken) {
      config.headers['Authorization'] = 'Bearer ' + getToken() // 让每个请求携带自定义token 请根据实际情况自行修改
    }
    // get请求映射params参数
    if (config.method === 'get' && config.params) {
      let url = config.url + '?' + tansParams(config.params)
      url = url.slice(0, -1)
      config.params = {}
      config.url = url
    }
    if (config.url.startsWith('/admin')) {
      config.baseURL = process.env.VUE_APP_ADMIN_API
    }
    if (
      !isRepeatSubmit &&
      (config.method === 'post' || config.method === 'put')
    ) {
      const requestObj = {
        url: config.url,
        time: new Date().getTime()
      }
      if (typeof config.data === 'object') {
        if (config.data.param) {
          let orgData = { orgcode: store.state.user.orgCode }
          //config.data.param.orgcode = "10";
          if (!config.data.param.orgcode) {
            Object.assign(config.data.param, orgData)
          }
          config.data.param = JSON.stringify(config.data.param)
        }
        requestObj.data = config.data
      } else {
        requestObj.data = config.data
      }
      const sessionObj = cache.session.getJSON('sessionObj')
      if (
        sessionObj === undefined ||
        sessionObj === null ||
        sessionObj === ''
      ) {
        cache.session.setJSON('sessionObj', requestObj)
      } else {
        const s_url = sessionObj.url // 请求地址
        const s_data = sessionObj.data // 请求数据
        const s_time = sessionObj.time // 请求时间
        const interval = 1000 // 间隔时间(ms)，小于此时间视为重复提交
        if (
          s_data === requestObj.data &&
          requestObj.time - s_time < interval &&
          s_url === requestObj.url
        ) {
          const message = '数据正在处理，请勿重复提交'
          console.warn(`[${s_url}]: ` + message)
          return Promise.reject(new Error(message))
        } else {
          cache.session.setJSON('sessionObj', requestObj)
        }
      }
    }
    return config
  },
  (error) => {
    console.log(error)
    Promise.reject(error)
  }
)

// 响应拦截器
service.interceptors.response.use(
  (res) => {
    // console.log('res',res)
    // 未设置状态码则默认成功状态
    const code = parseInt(res.data.code) || 200
    // 获取错误信息
    const msg = errorCode[code] || res.data.msg || errorCode['default']
    // 二进制数据则直接返回
    if (
      res.request.responseType === 'blob' ||
      res.request.responseType === 'arraybuffer'
    ) {
      return res.data
    }
    if (code === 401 || res.status === 403 || code === 403) {
      if (!isReloginShow) {
        isReloginShow = true
        MessageBox.confirm(
          '登录状态已过期，您可以继续留在该页面，或者重新登录',
          '系统提示',
          {
            confirmButtonText: '重新登录',
            cancelButtonText: '取消',
            type: 'warning'
          }
        )
          .then(() => {
            isReloginShow = false
            store.dispatch('LogOut').then(() => {
              if (!defaultSettings.casEnable) {
                location.href = `/login?redirect=${location.pathname}`
                //location.href = '/index'
              }

              // 如果是登录页面不需要重新加载
              // if (window.location.hash.indexOf("#/login") != 0) {
              //   location.href = "/index";
              // }
            })
          })
          .catch(() => {
            isReloginShow = false
          })
      }
      return Promise.reject('无效的会话，或者会话已过期，请重新登录。')
    } else if (code === 500) {
      Message({
        message: msg,
        type: 'error'
      })
      return Promise.reject(new Error(msg))
    } else if (code !== 200) {
      Notification.error({
        title: msg
      })
      return Promise.reject(new Error(msg))
    } else {
      return res.data
    }
  },
  (error) => {
    let { message } = error
    if (message == 'Network Error') {
      message = '后端接口连接异常'
    } else if (message.includes('timeout')) {
      message = '系统接口请求超时'
    } else if (message.includes('Request failed with status code')) {
      message = '系统接口' + message.substr(message.length - 3) + '异常'
    }
    Message({
      message: message,
      type: 'error',
      duration: 5 * 1000
    })
    if (error.response && error.response.status && error.response.status === 403) {
      console.log('权限系统403,进入登录页面')
      removeToken()
      location.href = `/login?redirect=${location.pathname}`
    }
    return Promise.reject(error)
  }
)

export function download(url, params, filename) {
  let fullPath = `/download?fileName=${url}&delete=true`
  return service.get(fullPath, {
    responseType: 'blob'
  }).then(async(data) => {
    let blob = new Blob([data])
    saveAs(blob, url)
  })
}

export function downloadExcelToPrint(url, params, filename) {
  let fullPath = `/download?fileName=${url}&delete=false`
  return service.get(fullPath, {
    responseType: 'blob'
  }).then(async(data) => {
    let blob = new Blob([data], { type: 'application/pdf' })
    let date = (new Date()).getTime()
    let ifr = document.createElement('iframe')
    ifr.style.frameborder = 'no'
    ifr.style.display = 'none'
    ifr.style.pageBreakBefore = 'always'
    ifr.setAttribute('id', 'printPdf' + date)
    ifr.setAttribute('name', 'printPdf' + date)
    ifr.src = window.URL.createObjectURL(blob)
    document.body.appendChild(ifr)
    const ordonnance = document.getElementById('printPdf' + date).contentWindow
    setTimeout(() => {
      ordonnance.print()
    })
    window.URL.revokeObjectURL(ifr.src) // 释放URL 对象
  })
}

// 通用下载方法
export function downloadTest(url, params, filename) {
  downloadLoadingInstance = Loading.service({
    text: '正在下载数据，请稍候',
    spinner: 'el-icon-loading',
    background: 'rgba(0, 0, 0, 0.7)'
  })
  return service
    .post(url, params, {
      transformRequest: [
        (params) => {
          return tansParams(params)
        }
      ],
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      responseType: 'blob'
    })
    .then(async(data) => {
      const isLogin = await blobValidate(data)
      if (isLogin) {
        const blob = new Blob([data])
        saveAs(blob, filename)
      } else {
        const resText = await data.text()
        const rspObj = JSON.parse(resText)
        const errMsg =
          errorCode[rspObj.code] || rspObj.msg || errorCode['default']
        Message.error(errMsg)
      }
      downloadLoadingInstance.close()
    })
    .catch((r) => {
      console.error(r)
      Message.error('下载文件出现错误，请联系管理员！')
      downloadLoadingInstance.close()
    })
}

export default service
