package com.wechat.framework.security.service;

import com.wechat.common.utils.IdUtils;
import com.wechat.common.utils.ServletUtils;
import com.wechat.common.utils.StringUtils;
import com.wechat.common.utils.ip.AddressUtils;
import com.wechat.common.utils.ip.IpUtils;
import com.wechat.framework.redis.RedisCache;
import com.wechat.framework.security.LoginUser;
import eu.bitwalker.useragentutils.UserAgent;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.servlet.http.HttpServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class TokenService {
   @Value("${token.header}")
   private String header;
   @Value("${token.secret}")
   private String secret;
   @Value("${token.expireTime}")
   private int expireTime;
   protected static final long MILLIS_SECOND = 1000L;
   protected static final long MILLIS_MINUTE = 60000L;
   private static final Long MILLIS_MINUTE_TEN = 1200000L;
   @Autowired
   private RedisCache redisCache;

   public LoginUser getLoginUser(HttpServletRequest request) {
      String token = this.getToken(request);
      if (StringUtils.isNotEmpty(token)) {
         Claims claims = this.parseToken(token);
         String uuid = (String)claims.get("login_user_key");
         String userKey = this.getTokenKey(uuid);
         return this.redisCache.getCacheObject(userKey);
      } else {
         return null;
      }
   }

   public String createToken(LoginUser loginUser) {
      String token = IdUtils.fastUUID();
      loginUser.setToken(token);
      this.setUserAgent(loginUser);
      this.refreshToken(loginUser);
      Map<String, Object> claims = new HashMap<>();
      claims.put("login_user_key", token);
      return this.createToken(claims);
   }

   public void verifyToken(LoginUser loginUser) {
      long expireTime = loginUser.getExpireTime();
      long currentTime = System.currentTimeMillis();
      if (expireTime - currentTime <= MILLIS_MINUTE_TEN) {
         String token = loginUser.getToken();
         loginUser.setToken(token);
         this.refreshToken(loginUser);
      }
   }

   public void refreshToken(LoginUser loginUser) {
      loginUser.setLoginTime(System.currentTimeMillis());
      loginUser.setExpireTime(loginUser.getLoginTime() + (long)this.expireTime * 60000L);
      String userKey = this.getTokenKey(loginUser.getToken());
      this.redisCache.setCacheObject(userKey, loginUser, this.expireTime, TimeUnit.MINUTES);
   }

   public void setUserAgent(LoginUser loginUser) {
      UserAgent userAgent = UserAgent.parseUserAgentString(ServletUtils.getRequest().getHeader("User-Agent"));
      String ip = IpUtils.getIpAddr(ServletUtils.getRequest());
      loginUser.setIpaddr(ip);
      loginUser.setLoginLocation(AddressUtils.getRealAddressByIP(ip));
      loginUser.setBrowser(userAgent.getBrowser().getName());
      loginUser.setOs(userAgent.getOperatingSystem().getName());
   }

   private String createToken(Map<String, Object> claims) {
      return Jwts.builder().setClaims(claims).signWith(SignatureAlgorithm.HS512, this.secret).compact();
   }

   private Claims parseToken(String token) {
      return (Claims)Jwts.parser().setSigningKey(this.secret).parseClaimsJws(token).getBody();
   }

   public String getUsernameFromToken(String token) {
      Claims claims = this.parseToken(token);
      return claims.getSubject();
   }

   private String getToken(HttpServletRequest request) {
      String token = request.getHeader(this.header);
      if (StringUtils.isNotEmpty(token) && token.startsWith("Bearer ")) {
         token = token.replace("Bearer ", "");
      }

      return token;
   }

   private String getTokenKey(String uuid) {
      return "login_tokens:" + uuid;
   }
}
