package com.wechat.project.system.controller;

import com.wechat.common.utils.SecurityUtils;
import com.wechat.framework.aspectj.lang.annotation.Log;
import com.wechat.framework.aspectj.lang.enums.BusinessType;
import com.wechat.framework.web.controller.BaseController;
import com.wechat.framework.web.domain.AjaxResult;
import com.wechat.project.system.domain.SysMenu;
import com.wechat.project.system.service.ISysMenuService;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping({"/system/menu"})
public class SysMenuController extends BaseController {
   @Autowired
   private ISysMenuService menuService;

   @PreAuthorize("@ss.hasPermi('system:menu:list')")
   @GetMapping({"/list"})
   public AjaxResult list(SysMenu menu) {
      List<SysMenu> menus = this.menuService.selectMenuList(menu);
      return AjaxResult.success(this.menuService.buildMenuTree(menus));
   }

   @PreAuthorize("@ss.hasPermi('system:menu:query')")
   @GetMapping({"/{menuId}"})
   public AjaxResult getInfo(@PathVariable Long menuId) {
      return AjaxResult.success(this.menuService.selectMenuById(menuId));
   }

   @PreAuthorize("@ss.hasPermi('system:menu:query')")
   @GetMapping({"/treeselect"})
   public AjaxResult treeselect(SysMenu dept) {
      List<SysMenu> menus = this.menuService.selectMenuList(dept);
      return AjaxResult.success(this.menuService.buildMenuTreeSelect(menus));
   }

   @PreAuthorize("@ss.hasPermi('system:menu:query')")
   @GetMapping({"/roleMenuTreeselect/{roleId}"})
   public AjaxResult roleMenuTreeselect(@PathVariable("roleId") Long roleId) {
      return AjaxResult.success(this.menuService.selectMenuListByRoleId(roleId));
   }

   @PreAuthorize("@ss.hasPermi('system:menu:add')")
   @Log(
      title = "菜单管理",
      businessType = BusinessType.INSERT
   )
   @PostMapping
   public AjaxResult add(@Validated @RequestBody SysMenu menu) {
      if ("1".equals(this.menuService.checkMenuNameUnique(menu))) {
         return AjaxResult.error("新增菜单'" + menu.getMenuName() + "'失败，菜单名称已存在");
      } else {
         menu.setCreateBy(SecurityUtils.getUsername());
         return this.toAjax(this.menuService.insertMenu(menu));
      }
   }

   @PreAuthorize("@ss.hasPermi('system:menu:edit')")
   @Log(
      title = "菜单管理",
      businessType = BusinessType.UPDATE
   )
   @PutMapping
   public AjaxResult edit(@Validated @RequestBody SysMenu menu) {
      if ("1".equals(this.menuService.checkMenuNameUnique(menu))) {
         return AjaxResult.error("修改菜单'" + menu.getMenuName() + "'失败，菜单名称已存在");
      } else {
         menu.setUpdateBy(SecurityUtils.getUsername());
         return this.toAjax(this.menuService.updateMenu(menu));
      }
   }

   @PreAuthorize("@ss.hasPermi('system:menu:remove')")
   @Log(
      title = "菜单管理",
      businessType = BusinessType.DELETE
   )
   @DeleteMapping({"/{menuId}"})
   public AjaxResult remove(@PathVariable("menuId") Long menuId) {
      if (this.menuService.hasChildByMenuId(menuId)) {
         return AjaxResult.error("存在子菜单,不允许删除");
      } else {
         return this.menuService.checkMenuExistRole(menuId) ? AjaxResult.error("菜单已分配,不允许删除") : this.toAjax(this.menuService.deleteMenuById(menuId));
      }
   }
}
