package com.wechat.project.system.service.impl;

import com.wechat.common.exception.CustomException;
import com.wechat.common.utils.StringUtils;
import com.wechat.framework.aspectj.lang.annotation.DataScope;
import com.wechat.project.system.domain.SysPost;
import com.wechat.project.system.domain.SysRole;
import com.wechat.project.system.domain.SysUser;
import com.wechat.project.system.domain.SysUserPost;
import com.wechat.project.system.domain.SysUserRole;
import com.wechat.project.system.mapper.SysPostMapper;
import com.wechat.project.system.mapper.SysRoleMapper;
import com.wechat.project.system.mapper.SysUserMapper;
import com.wechat.project.system.mapper.SysUserPostMapper;
import com.wechat.project.system.mapper.SysUserRoleMapper;
import com.wechat.project.system.service.ISysUserService;
import java.util.ArrayList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.PathVariable;

@Service
public class SysUserServiceImpl implements ISysUserService {
   @Autowired
   private SysUserMapper userMapper;
   @Autowired
   private SysRoleMapper roleMapper;
   @Autowired
   private SysPostMapper postMapper;
   @Autowired
   private SysUserRoleMapper userRoleMapper;
   @Autowired
   private SysUserPostMapper userPostMapper;

   @DataScope(
      deptAlias = "d",
      userAlias = "u"
   )
   @Override
   public List<SysUser> selectUserList(SysUser user) {
      return this.userMapper.selectUserList(user);
   }

   @Override
   public SysUser selectUserByUserName(String userName) {
      return this.userMapper.selectUserByUserName(userName);
   }

   @Override
   public SysUser selectUserById(Long userId) {
      return this.userMapper.selectUserById(userId);
   }

   @Override
   public String selectUserRoleGroup(String userName) {
      List<SysRole> list = this.roleMapper.selectRolesByUserName(userName);
      StringBuffer idsStr = new StringBuffer();

      for (SysRole role : list) {
         idsStr.append(role.getRoleName()).append(",");
      }

      return StringUtils.isNotEmpty(idsStr.toString()) ? idsStr.substring(0, idsStr.length() - 1) : idsStr.toString();
   }

   @Override
   public String selectUserPostGroup(String userName) {
      List<SysPost> list = this.postMapper.selectPostsByUserName(userName);
      StringBuffer idsStr = new StringBuffer();

      for (SysPost post : list) {
         idsStr.append(post.getPostName()).append(",");
      }

      return StringUtils.isNotEmpty(idsStr.toString()) ? idsStr.substring(0, idsStr.length() - 1) : idsStr.toString();
   }

   @Override
   public String checkUserNameUnique(String userName) {
      int count = this.userMapper.checkUserNameUnique(userName);
      return count > 0 ? "1" : "0";
   }

   @Override
   public String checkPhoneUnique(SysUser user) {
      Long userId = StringUtils.isNull(user.getUserId()) ? -1L : user.getUserId();
      SysUser info = this.userMapper.checkPhoneUnique(user.getPhonenumber());
      return StringUtils.isNotNull(info) && info.getUserId() != userId ? "1" : "0";
   }

   @Override
   public String checkEmailUnique(SysUser user) {
      Long userId = StringUtils.isNull(user.getUserId()) ? -1L : user.getUserId();
      SysUser info = this.userMapper.checkEmailUnique(user.getEmail());
      return StringUtils.isNotNull(info) && info.getUserId() != userId ? "1" : "0";
   }

   @Override
   public void checkUserAllowed(SysUser user) {
      if (StringUtils.isNotNull(user.getUserId()) && user.isAdmin()) {
         throw new CustomException("不允许操作超级管理员用户");
      }
   }

   @Transactional
   @Override
   public int insertUser(SysUser user) {
      int rows = this.userMapper.insertUser(user);
      this.insertUserPost(user);
      this.insertUserRole(user);
      return rows;
   }

   @Transactional
   @Override
   public int updateUser(SysUser user) {
      Long userId = user.getUserId();
      this.userRoleMapper.deleteUserRoleByUserId(userId);
      this.insertUserRole(user);
      this.userPostMapper.deleteUserPostByUserId(userId);
      this.insertUserPost(user);
      return this.userMapper.updateUser(user);
   }

   @Override
   public int updateUserStatus(SysUser user) {
      return this.userMapper.updateUser(user);
   }

   @Override
   public int updateUserProfile(SysUser user) {
      return this.userMapper.updateUser(user);
   }

   @Override
   public boolean updateUserAvatar(String userName, String avatar) {
      return this.userMapper.updateUserAvatar(userName, avatar) > 0;
   }

   @Override
   public int resetPwd(SysUser user) {
      return this.userMapper.updateUser(user);
   }

   @Override
   public int resetUserPwd(String userName, String password) {
      return this.userMapper.resetUserPwd(userName, password);
   }

   public void insertUserRole(SysUser user) {
      Long[] roles = user.getRoleIds();
      if (StringUtils.isNotNull(roles)) {
         List<SysUserRole> list = new ArrayList<>();

         for (Long roleId : roles) {
            SysUserRole ur = new SysUserRole();
            ur.setUserId(user.getUserId());
            ur.setRoleId(roleId);
            list.add(ur);
         }

         if (list.size() > 0) {
            this.userRoleMapper.batchUserRole(list);
         }
      }
   }

   public void insertUserPost(SysUser user) {
      Long[] posts = user.getPostIds();
      if (StringUtils.isNotNull(posts)) {
         List<SysUserPost> list = new ArrayList<>();

         for (Long postId : posts) {
            SysUserPost up = new SysUserPost();
            up.setUserId(user.getUserId());
            up.setPostId(postId);
            list.add(up);
         }

         if (list.size() > 0) {
            this.userPostMapper.batchUserPost(list);
         }
      }
   }

   @Override
   public int deleteUserById(Long userId) {
      this.userRoleMapper.deleteUserRoleByUserId(userId);
      this.userPostMapper.deleteUserPostByUserId(userId);
      return this.userMapper.deleteUserById(userId);
   }

   @Override
   public int deleteUserByIds(Long[] userIds) {
      for (Long userId : userIds) {
         this.checkUserAllowed(new SysUser(userId));
      }

      return this.userMapper.deleteUserByIds(userIds);
   }

   @Override
   public SysUser getUserinfo(@PathVariable String username, @PathVariable String password) {
      return this.userMapper.getUserinfo(username, password);
   }
}
