package com.wechat.framework.aspectj;

import com.alibaba.fastjson.JSON;
import com.wechat.common.enums.HttpMethod;
import com.wechat.common.utils.ServletUtils;
import com.wechat.common.utils.StringUtils;
import com.wechat.common.utils.ip.IpUtils;
import com.wechat.common.utils.spring.SpringUtils;
import com.wechat.framework.aspectj.lang.annotation.Log;
import com.wechat.framework.aspectj.lang.enums.BusinessStatus;
import com.wechat.framework.manager.AsyncManager;
import com.wechat.framework.manager.factory.AsyncFactory;
import com.wechat.framework.security.LoginUser;
import com.wechat.framework.security.service.TokenService;
import com.wechat.project.monitor.domain.SysOperLog;
import java.lang.reflect.Method;
import java.util.Map;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.HandlerMapping;

@Aspect
@Component
public class LogAspect {
   private static final Logger log = LoggerFactory.getLogger(LogAspect.class);

   @Pointcut("@annotation(com.wechat.framework.aspectj.lang.annotation.Log)")
   public void logPointCut() {
   }

   @AfterReturning(
      pointcut = "logPointCut()",
      returning = "jsonResult"
   )
   public void doAfterReturning(JoinPoint joinPoint, Object jsonResult) {
      this.handleLog(joinPoint, null, jsonResult);
   }

   @AfterThrowing(
      value = "logPointCut()",
      throwing = "e"
   )
   public void doAfterThrowing(JoinPoint joinPoint, Exception e) {
      this.handleLog(joinPoint, e, null);
   }

   protected void handleLog(final JoinPoint joinPoint, final Exception e, Object jsonResult) {
      try {
         Log controllerLog = this.getAnnotationLog(joinPoint);
         if (controllerLog == null) {
            return;
         }

         LoginUser loginUser = SpringUtils.getBean(TokenService.class).getLoginUser(ServletUtils.getRequest());
         SysOperLog operLog = new SysOperLog();
         operLog.setStatus(BusinessStatus.SUCCESS.ordinal());
         String ip = IpUtils.getIpAddr(ServletUtils.getRequest());
         operLog.setOperIp(ip);
         operLog.setJsonResult(JSON.toJSONString(jsonResult));
         operLog.setOperUrl(ServletUtils.getRequest().getRequestURI());
         if (loginUser != null) {
            operLog.setOperName(loginUser.getUsername());
         }

         if (e != null) {
            operLog.setStatus(BusinessStatus.FAIL.ordinal());
            operLog.setErrorMsg(StringUtils.substring(e.getMessage(), 0, 2000));
         }

         String className = joinPoint.getTarget().getClass().getName();
         String methodName = joinPoint.getSignature().getName();
         operLog.setMethod(className + "." + methodName + "()");
         operLog.setRequestMethod(ServletUtils.getRequest().getMethod());
         this.getControllerMethodDescription(joinPoint, controllerLog, operLog);
         AsyncManager.me().execute(AsyncFactory.recordOper(operLog));
      } catch (Exception var10) {
         log.error("==前置通知异常==");
         log.error("异常信息:{}", var10.getMessage());
         var10.printStackTrace();
      }
   }

   public void getControllerMethodDescription(JoinPoint joinPoint, Log log, SysOperLog operLog) throws Exception {
      operLog.setBusinessType(log.businessType().ordinal());
      operLog.setTitle(log.title());
      operLog.setOperatorType(log.operatorType().ordinal());
      if (log.isSaveRequestData()) {
         this.setRequestValue(joinPoint, operLog);
      }
   }

   private void setRequestValue(JoinPoint joinPoint, SysOperLog operLog) throws Exception {
      String requestMethod = operLog.getRequestMethod();
      if (!HttpMethod.PUT.name().equals(requestMethod) && !HttpMethod.POST.name().equals(requestMethod)) {
         Map<?, ?> paramsMap = (Map<?, ?>)ServletUtils.getRequest().getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
         operLog.setOperParam(StringUtils.substring(paramsMap.toString(), 0, 2000));
      } else {
         String params = this.argsArrayToString(joinPoint.getArgs());
         operLog.setOperParam(StringUtils.substring(params, 0, 2000));
      }
   }

   private Log getAnnotationLog(JoinPoint joinPoint) throws Exception {
      Signature signature = joinPoint.getSignature();
      MethodSignature methodSignature = (MethodSignature)signature;
      Method method = methodSignature.getMethod();
      return method != null ? method.getAnnotation(Log.class) : null;
   }

   private String argsArrayToString(Object[] paramsArray) {
      String params = "";
      if (paramsArray != null && paramsArray.length > 0) {
         for (int i = 0; i < paramsArray.length; i++) {
            if (!(paramsArray[i] instanceof MultipartFile)) {
               Object jsonObj = JSON.toJSON(paramsArray[i]);
               params = params + jsonObj.toString() + " ";
            }
         }
      }

      return params.trim();
   }
}
