package com.wechat.project.system.controller;

import com.wechat.common.utils.SecurityUtils;
import com.wechat.common.utils.poi.ExcelUtil;
import com.wechat.framework.aspectj.lang.annotation.Log;
import com.wechat.framework.aspectj.lang.enums.BusinessType;
import com.wechat.framework.web.controller.BaseController;
import com.wechat.framework.web.domain.AjaxResult;
import com.wechat.framework.web.page.TableDataInfo;
import com.wechat.project.system.domain.SysUser;
import com.wechat.project.system.service.ISysPostService;
import com.wechat.project.system.service.ISysRoleService;
import com.wechat.project.system.service.ISysUserService;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping({"/system/user"})
public class SysUserController extends BaseController {
   @Autowired
   private ISysUserService userService;
   @Autowired
   private ISysRoleService roleService;
   @Autowired
   private ISysPostService postService;

   @PreAuthorize("@ss.hasPermi('system:user:list')")
   @GetMapping({"/list"})
   public TableDataInfo list(SysUser user) {
      this.startPage();
      List<SysUser> list = this.userService.selectUserList(user);
      return this.getDataTable(list);
   }

   @Log(
      title = "用户管理",
      businessType = BusinessType.EXPORT
   )
   @PreAuthorize("@ss.hasPermi('system:user:export')")
   @GetMapping({"/export"})
   public AjaxResult export(SysUser user) {
      List<SysUser> list = this.userService.selectUserList(user);
      ExcelUtil<SysUser> util = new ExcelUtil<>(SysUser.class);
      return util.exportExcel(list, "用户数据");
   }

   @PreAuthorize("@ss.hasPermi('system:user:query')")
   @GetMapping({"/{userId}"})
   public AjaxResult getInfo(@PathVariable Long userId) {
      AjaxResult ajax = AjaxResult.success(this.userService.selectUserById(userId));
      ajax.put("postIds", this.postService.selectPostListByUserId(userId));
      ajax.put("roleIds", this.roleService.selectRoleListByUserId(userId));
      return ajax;
   }

   @PreAuthorize("@ss.hasPermi('system:user:add')")
   @Log(
      title = "用户管理",
      businessType = BusinessType.INSERT
   )
   @PostMapping
   public AjaxResult add(@Validated @RequestBody SysUser user) {
      if ("1".equals(this.userService.checkUserNameUnique(user.getUserName()))) {
         return AjaxResult.error("新增用户'" + user.getUserName() + "'失败，登录账号已存在");
      } else if ("1".equals(this.userService.checkPhoneUnique(user))) {
         return AjaxResult.error("新增用户'" + user.getUserName() + "'失败，手机号码已存在");
      } else if ("1".equals(this.userService.checkEmailUnique(user))) {
         return AjaxResult.error("新增用户'" + user.getUserName() + "'失败，邮箱账号已存在");
      } else {
         user.setCreateBy(SecurityUtils.getUsername());
         user.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
         return this.toAjax(this.userService.insertUser(user));
      }
   }

   @PreAuthorize("@ss.hasPermi('system:user:edit')")
   @Log(
      title = "用户管理",
      businessType = BusinessType.UPDATE
   )
   @PutMapping
   public AjaxResult edit(@Validated @RequestBody SysUser user) {
      this.userService.checkUserAllowed(user);
      if ("1".equals(this.userService.checkPhoneUnique(user))) {
         return AjaxResult.error("修改用户'" + user.getUserName() + "'失败，手机号码已存在");
      } else if ("1".equals(this.userService.checkEmailUnique(user))) {
         return AjaxResult.error("修改用户'" + user.getUserName() + "'失败，邮箱账号已存在");
      } else {
         user.setUpdateBy(SecurityUtils.getUsername());
         return this.toAjax(this.userService.updateUser(user));
      }
   }

   @PreAuthorize("@ss.hasPermi('system:user:remove')")
   @Log(
      title = "用户管理",
      businessType = BusinessType.DELETE
   )
   @DeleteMapping({"/{userIds}"})
   public AjaxResult remove(@PathVariable Long[] userIds) {
      return this.toAjax(this.userService.deleteUserByIds(userIds));
   }

   @PreAuthorize("@ss.hasPermi('system:user:edit')")
   @Log(
      title = "用户管理",
      businessType = BusinessType.UPDATE
   )
   @PutMapping({"/resetPwd"})
   public AjaxResult resetPwd(@RequestBody SysUser user) {
      this.userService.checkUserAllowed(user);
      user.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
      user.setUpdateBy(SecurityUtils.getUsername());
      return this.toAjax(this.userService.resetPwd(user));
   }

   @PreAuthorize("@ss.hasPermi('system:user:edit')")
   @Log(
      title = "用户管理",
      businessType = BusinessType.UPDATE
   )
   @PutMapping({"/changeStatus"})
   public AjaxResult changeStatus(@RequestBody SysUser user) {
      this.userService.checkUserAllowed(user);
      user.setUpdateBy(SecurityUtils.getUsername());
      return this.toAjax(this.userService.updateUserStatus(user));
   }
}
