package com.wechat.project.system.service.impl;

import com.wechat.common.exception.CustomException;
import com.wechat.common.utils.ServletUtils;
import com.wechat.common.utils.StringUtils;
import com.wechat.common.utils.spring.SpringUtils;
import com.wechat.framework.aspectj.lang.annotation.DataScope;
import com.wechat.framework.security.LoginUser;
import com.wechat.framework.security.service.TokenService;
import com.wechat.framework.web.domain.TreeSelect;
import com.wechat.project.system.domain.SysDept;
import com.wechat.project.system.domain.SysDeptChannel;
import com.wechat.project.system.mapper.SysDeptChannelMapper;
import com.wechat.project.system.mapper.SysDeptMapper;
import com.wechat.project.system.service.ISysDeptService;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
public class SysDeptServiceImpl implements ISysDeptService {
   @Autowired
   private SysDeptMapper deptMapper;
   @Autowired
   private SysDeptChannelMapper deptChannelMapper;

   @DataScope(
      deptAlias = "d"
   )
   @Override
   public List<SysDept> selectDeptList(SysDept dept) {
      LoginUser currentUser = SpringUtils.getBean(TokenService.class).getLoginUser(ServletUtils.getRequest());
      if (currentUser != null && !currentUser.getUsername().equals("admin")) {
         Long deptid = currentUser.getDeptId();
         dept.setDeptId(deptid);
      }

      return this.deptMapper.selectDeptList(dept);
   }

   @Override
   public List<SysDept> selectDeptInfo() {
      return this.deptMapper.selectDeptInfo();
   }

   @Override
   public List<SysDept> getHspOption() {
      return this.deptMapper.getHspOption();
   }

   @Override
   public List<SysDept> getDeptOption(SysDept deptInfo) {
      return this.deptMapper.getDeptOption(deptInfo);
   }

   @Override
   public List<SysDept> buildDeptTree(List<SysDept> depts) {
      List<SysDept> returnList = new ArrayList<>();

      for (SysDept t : depts) {
         if (t.getParentId() == 0L) {
            this.recursionFn(depts, t);
            returnList.add(t);
         }
      }

      if (returnList.isEmpty()) {
         returnList = depts;
      }

      return returnList;
   }

   @Override
   public List<TreeSelect> buildDeptTreeSelect(List<SysDept> depts) {
      List<SysDept> deptTrees = this.buildDeptTree(depts);
      return deptTrees.stream().map(TreeSelect::new).collect(Collectors.toList());
   }

   @Override
   public List<Integer> selectDeptListByRoleId(Long roleId) {
      return this.deptMapper.selectDeptListByRoleId(roleId);
   }

   @Override
   public SysDept selectDeptById(Long deptId) {
      return this.deptMapper.selectDeptById(deptId);
   }

   @Override
   public boolean hasChildByDeptId(Long deptId) {
      int result = this.deptMapper.hasChildByDeptId(deptId);
      return result > 0;
   }

   @Override
   public boolean checkDeptExistUser(Long deptId) {
      int result = this.deptMapper.checkDeptExistUser(deptId);
      return result > 0;
   }

   @Override
   public String checkDeptNameUnique(SysDept dept) {
      Long deptId = StringUtils.isNull(dept.getDeptId()) ? -1L : dept.getDeptId();
      SysDept info = this.deptMapper.checkDeptNameUnique(dept.getDeptName(), dept.getParentId());
      return StringUtils.isNotNull(info) && info.getDeptId() != deptId ? "1" : "0";
   }

   @Override
   public int insertDept(SysDept dept) {
      SysDept info = this.deptMapper.selectDeptById(dept.getParentId());
      if (!"0".equals(info.getStatus())) {
         throw new CustomException("部门停用，不允许新增");
      } else {
         dept.setAncestors(info.getAncestors() + "," + dept.getParentId());
         this.deptMapper.insertDept(dept);
         return this.insertDeptChannel(dept);
      }
   }

   public int insertDeptChannel(SysDept dept) {
      int rows = 1;
      List<SysDeptChannel> list = Arrays.stream(dept.getChannelIds())
         .peek(channelId -> System.out.println(dept.getDeptId()))
         .map(channelId -> new SysDeptChannel(dept.getDeptId(), channelId))
         .collect(Collectors.toList());
      if (list.size() > 0) {
         rows = this.deptChannelMapper.batchDeptChannel(list);
      }

      return rows;
   }

   @Transactional
   @Override
   public int updateDept(SysDept dept) {
      SysDept newParentDept = this.deptMapper.selectDeptById(dept.getParentId());
      SysDept oldDept = this.deptMapper.selectDeptById(dept.getDeptId());
      if (StringUtils.isNotNull(newParentDept) && StringUtils.isNotNull(oldDept)) {
         String newAncestors = newParentDept.getAncestors() + "," + newParentDept.getDeptId();
         String oldAncestors = oldDept.getAncestors();
         dept.setAncestors(newAncestors);
         this.updateDeptChildren(dept.getDeptId(), newAncestors, oldAncestors);
      }

      int result = this.deptMapper.updateDept(dept);
      if ("0".equals(dept.getStatus())) {
         this.updateParentDeptStatus(dept);
      }

      this.deptChannelMapper.deleteChannelByDeptId(dept.getDeptId());
      return this.insertDeptChannel(dept);
   }

   private void updateParentDeptStatus(SysDept dept) {
      String updateBy = dept.getUpdateBy();
      dept = this.deptMapper.selectDeptById(dept.getDeptId());
      dept.setUpdateBy(updateBy);
      this.deptMapper.updateDeptStatus(dept);
   }

   public void updateDeptChildren(Long deptId, String newAncestors, String oldAncestors) {
      List<SysDept> children = this.deptMapper.selectChildrenDeptById(deptId);

      for (SysDept child : children) {
         child.setAncestors(child.getAncestors().replace(oldAncestors, newAncestors));
      }

      if (children.size() > 0) {
         this.deptMapper.updateDeptChildren(children);
      }
   }

   @Override
   public int deleteDeptById(Long deptId) {
      return this.deptMapper.deleteDeptById(deptId);
   }

   private void recursionFn(List<SysDept> list, SysDept t) {
      List<SysDept> childList = this.getChildList(list, t);
      t.setChildren(childList);

      for (SysDept tChild : childList) {
         if (this.hasChild(list, tChild)) {
            for (SysDept n : childList) {
               this.recursionFn(list, n);
            }
         }
      }
   }

   private List<SysDept> getChildList(List<SysDept> list, SysDept t) {
      List<SysDept> tlist = new ArrayList<>();

      for (SysDept n : list) {
         if (n.getParentId() == t.getDeptId()) {
            tlist.add(n);
         }
      }

      return tlist;
   }

   private boolean hasChild(List<SysDept> list, SysDept t) {
      return this.getChildList(list, t).size() > 0;
   }
}
