package com.wechat.project.system.service.impl;

import com.wechat.common.exception.CustomException;
import com.wechat.common.utils.StringUtils;
import com.wechat.framework.aspectj.lang.annotation.DataScope;
import com.wechat.project.system.domain.SysRole;
import com.wechat.project.system.domain.SysRoleDept;
import com.wechat.project.system.domain.SysRoleMenu;
import com.wechat.project.system.mapper.SysRoleDeptMapper;
import com.wechat.project.system.mapper.SysRoleMapper;
import com.wechat.project.system.mapper.SysRoleMenuMapper;
import com.wechat.project.system.mapper.SysUserRoleMapper;
import com.wechat.project.system.service.ISysRoleService;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
public class SysRoleServiceImpl implements ISysRoleService {
   @Autowired
   private SysRoleMapper roleMapper;
   @Autowired
   private SysRoleMenuMapper roleMenuMapper;
   @Autowired
   private SysUserRoleMapper userRoleMapper;
   @Autowired
   private SysRoleDeptMapper roleDeptMapper;

   @DataScope(
      deptAlias = "d"
   )
   @Override
   public List<SysRole> selectRoleList(SysRole role) {
      return this.roleMapper.selectRoleList(role);
   }

   @Override
   public Set<String> selectRolePermissionByUserId(Long userId) {
      List<SysRole> perms = this.roleMapper.selectRolePermissionByUserId(userId);
      Set<String> permsSet = new HashSet<>();

      for (SysRole perm : perms) {
         if (StringUtils.isNotNull(perm)) {
            permsSet.addAll(Arrays.asList(perm.getRoleKey().trim().split(",")));
         }
      }

      return permsSet;
   }

   @Override
   public List<SysRole> selectRoleAll() {
      return this.roleMapper.selectRoleAll();
   }

   @Override
   public List<Integer> selectRoleListByUserId(Long userId) {
      return this.roleMapper.selectRoleListByUserId(userId);
   }

   @Override
   public SysRole selectRoleById(Long roleId) {
      return this.roleMapper.selectRoleById(roleId);
   }

   @Override
   public String checkRoleNameUnique(SysRole role) {
      Long roleId = StringUtils.isNull(role.getRoleId()) ? -1L : role.getRoleId();
      SysRole info = this.roleMapper.checkRoleNameUnique(role.getRoleName());
      return StringUtils.isNotNull(info) && info.getRoleId() != roleId ? "1" : "0";
   }

   @Override
   public String checkRoleKeyUnique(SysRole role) {
      Long roleId = StringUtils.isNull(role.getRoleId()) ? -1L : role.getRoleId();
      SysRole info = this.roleMapper.checkRoleKeyUnique(role.getRoleKey());
      return StringUtils.isNotNull(info) && info.getRoleId() != roleId ? "1" : "0";
   }

   @Override
   public void checkRoleAllowed(SysRole role) {
      if (StringUtils.isNotNull(role.getRoleId()) && role.isAdmin()) {
         throw new CustomException("不允许操作超级管理员角色");
      }
   }

   @Override
   public int countUserRoleByRoleId(Long roleId) {
      return this.userRoleMapper.countUserRoleByRoleId(roleId);
   }

   @Transactional
   @Override
   public int insertRole(SysRole role) {
      this.roleMapper.insertRole(role);
      return this.insertRoleMenu(role);
   }

   @Transactional
   @Override
   public int updateRole(SysRole role) {
      this.roleMapper.updateRole(role);
      this.roleMenuMapper.deleteRoleMenuByRoleId(role.getRoleId());
      return this.insertRoleMenu(role);
   }

   @Override
   public int updateRoleStatus(SysRole role) {
      return this.roleMapper.updateRole(role);
   }

   @Transactional
   @Override
   public int authDataScope(SysRole role) {
      this.roleMapper.updateRole(role);
      this.roleDeptMapper.deleteRoleDeptByRoleId(role.getRoleId());
      return this.insertRoleDept(role);
   }

   public int insertRoleMenu(SysRole role) {
      int rows = 1;
      List<SysRoleMenu> list = new ArrayList<>();

      for (Long menuId : role.getMenuIds()) {
         SysRoleMenu rm = new SysRoleMenu();
         rm.setRoleId(role.getRoleId());
         rm.setMenuId(menuId);
         list.add(rm);
      }

      if (list.size() > 0) {
         rows = this.roleMenuMapper.batchRoleMenu(list);
      }

      return rows;
   }

   public int insertRoleDept(SysRole role) {
      int rows = 1;
      List<SysRoleDept> list = new ArrayList<>();

      for (Long deptId : role.getDeptIds()) {
         SysRoleDept rd = new SysRoleDept();
         rd.setRoleId(role.getRoleId());
         rd.setDeptId(deptId);
         list.add(rd);
      }

      if (list.size() > 0) {
         rows = this.roleDeptMapper.batchRoleDept(list);
      }

      return rows;
   }

   @Override
   public int deleteRoleById(Long roleId) {
      return this.roleMapper.deleteRoleById(roleId);
   }

   @Override
   public int deleteRoleByIds(Long[] roleIds) {
      for (Long roleId : roleIds) {
         this.checkRoleAllowed(new SysRole(roleId));
         SysRole role = this.selectRoleById(roleId);
         if (this.countUserRoleByRoleId(roleId) > 0) {
            throw new CustomException(String.format("%1$s已分配,不能删除", role.getRoleName()));
         }
      }

      return this.roleMapper.deleteRoleByIds(roleIds);
   }
}
