package com.wechat.project.system.controller;

import com.wechat.common.utils.SecurityUtils;
import com.wechat.framework.aspectj.lang.annotation.Log;
import com.wechat.framework.aspectj.lang.enums.BusinessType;
import com.wechat.framework.web.controller.BaseController;
import com.wechat.framework.web.domain.AjaxResult;
import com.wechat.project.system.domain.SysDept;
import com.wechat.project.system.service.ISysDeptService;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping({"/system/dept"})
public class SysDeptController extends BaseController {
   @Autowired
   private ISysDeptService deptService;

   @PreAuthorize("@ss.hasPermi('system:dept:list')")
   @GetMapping({"/list"})
   public AjaxResult list(SysDept dept) {
      List<SysDept> depts = this.deptService.selectDeptList(dept);
      return AjaxResult.success(this.deptService.buildDeptTree(depts));
   }

   @PreAuthorize("@ss.hasPermi('system:dept:list')")
   @GetMapping({"/getHspOption"})
   public AjaxResult getHspOption() {
      List<SysDept> hspList = this.deptService.selectDeptList(new SysDept());
      return AjaxResult.success(hspList);
   }

   @PreAuthorize("@ss.hasPermi('system:dept:list')")
   @GetMapping({"/getDeptOption"})
   public AjaxResult getDeptOption(SysDept deptInfo) {
      deptInfo.setDelFlag("0");
      deptInfo.setStatus("0");
      List<SysDept> list = this.deptService.getDeptOption(deptInfo);
      return AjaxResult.success(this.deptService.buildDeptTreeSelect(list));
   }

   @PreAuthorize("@ss.hasPermi('system:dept:query')")
   @GetMapping({"/{deptId}"})
   public AjaxResult getInfo(@PathVariable Long deptId) {
      return AjaxResult.success(this.deptService.selectDeptById(deptId));
   }

   @PreAuthorize("@ss.hasPermi('system:dept:query')")
   @GetMapping({"/treeselect"})
   public AjaxResult treeselect(SysDept dept) {
      List<SysDept> depts = this.deptService.selectDeptList(dept);
      return AjaxResult.success(this.deptService.buildDeptTreeSelect(depts));
   }

   @PreAuthorize("@ss.hasPermi('system:dept:query')")
   @GetMapping({"/roleDeptTreeselect/{roleId}"})
   public AjaxResult roleDeptTreeselect(@PathVariable("roleId") Long roleId) {
      return AjaxResult.success(this.deptService.selectDeptListByRoleId(roleId));
   }

   @PreAuthorize("@ss.hasPermi('system:dept:add')")
   @Log(
      title = "部门管理",
      businessType = BusinessType.INSERT
   )
   @PostMapping
   public AjaxResult add(@Validated @RequestBody SysDept dept) {
      if ("1".equals(this.deptService.checkDeptNameUnique(dept))) {
         return AjaxResult.error("新增部门'" + dept.getDeptName() + "'失败，部门名称已存在");
      } else {
         dept.setCreateBy(SecurityUtils.getUsername());
         return this.toAjax(this.deptService.insertDept(dept));
      }
   }

   @PreAuthorize("@ss.hasPermi('system:dept:edit')")
   @Log(
      title = "部门管理",
      businessType = BusinessType.UPDATE
   )
   @PutMapping
   public AjaxResult edit(@Validated @RequestBody SysDept dept) {
      if ("1".equals(this.deptService.checkDeptNameUnique(dept))) {
         return AjaxResult.error("修改部门'" + dept.getDeptName() + "'失败，部门名称已存在");
      } else if (dept.getParentId().equals(dept.getDeptId())) {
         return AjaxResult.error("修改部门'" + dept.getDeptName() + "'失败，上级部门不能是自己");
      } else {
         dept.setUpdateBy(SecurityUtils.getUsername());
         return this.toAjax(this.deptService.updateDept(dept));
      }
   }

   @PreAuthorize("@ss.hasPermi('system:dept:remove')")
   @Log(
      title = "部门管理",
      businessType = BusinessType.DELETE
   )
   @DeleteMapping({"/{deptId}"})
   public AjaxResult remove(@PathVariable Long deptId) {
      if (this.deptService.hasChildByDeptId(deptId)) {
         return AjaxResult.error("存在下级部门,不允许删除");
      } else {
         return this.deptService.checkDeptExistUser(deptId) ? AjaxResult.error("部门存在用户,不允许删除") : this.toAjax(this.deptService.deleteDeptById(deptId));
      }
   }
}
